/*
 * LibGGI3D 
 * 
 * Phong lighting model shader
 * 
 * Copyright (C) 1999 Jon Taylor [taylorj@ggi-project.org]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <math.h>
#include <ggi/ggi.h>
#include <glib.h>
#include <meschach/matrix.h>
#include <ggi3d/modules/globals.h>
#include <ggi3d/modules/phong_light.h>

char identstr_phong_light[] = "Phong lighting model 0.1";

/* We have no private data yet */
typedef struct 
{
	unsigned int junk;
} phong_light_private;

ggi3d_modp ggi3d_dlinit()
{
	ggi3d_modp modinfo;
	phong_light_private *priv;

	modinfo = malloc(sizeof(ggi3d_module));
	modinfo->destroy = phong_light_destroy;
	modinfo->handle_input = phong_light_handle;
	modinfo->command = phong_light_command;
	
	priv = malloc(sizeof(phong_light_private));
	modinfo->priv = priv;
	
	return modinfo;
}


void calculate_local_shade(GList *the_material,
			   VEC *the_surface_normal,
			   GList *the_lights,
			   ggi_color *ambient_light_intensity,
			   VEC *vector_to_eye);

/* Free the private data */
void phong_light_destroy(ggi3d_modp self)
{
	free(self->priv);
	return;
}

int phong_light_command(ggi3d_modp self, int cmd, void *data_in, void *data_out)
{

	switch (cmd)
	{
		case CMD_GET_IDENT:
		data_out = (void *)identstr_phong_light;
		return 1;
		break;
	}
	return 0;
}

int phong_light_handle(ggi3d_modp self, void *data_in, void *data_out)
{
//	phong_light_private *priv=(phong_light_private *)self->priv;
	/* We currently don't make use of the private data field */

//	calculate_local_shade((world_data *)data_in);

	return 0; /* We always handle all input in one gulp */
}

/* calculate_local_shade(): Applies the Phong reflection model to compute
** the shade of a pixel given the light sources in the world.
*/
void calculate_local_shade(GList *the_material,
			   VEC *the_surface_normal,
			   GList *the_lights,
			   ggi_color *ambient_light_intensity,
			   VEC *vector_to_eye)
{
	VEC *h, *t;
	GList *light_ptr;
	light *the_light;
	float sum_red, sum_green, sum_blue;
	material *my_material;
	
	float cos_theta, cosn_alpha, tempca, hold, check;
	float diffuse_red, diffuse_green, diffuse_blue;
	float specular_red, specular_green, specular_blue;
	int n, i;
	
	my_material = (material *)the_material->data;

	/* Initialize running total summation components */
	sum_red = 0.0;
	sum_green = 0.0;
	sum_blue = 0.0;
	
	/* Sum over all light sources in the light_list */
	light_ptr = the_lst;
	while (light_ptr != g_list_last(light_ptr))
	{
		the_light = (light *)light_ptr->data;
		
		/* Calculate cos_theta, the cosine of theta.  Theta is is the angle 
		 between the surface normal and the light vector. */
		//		cos_theta = unit_vector(the_surface_normal) * 
		//		  ((unit_vector(the_light.the_direction)) * (-1.0));
		
		/* If cos_theta is less than zero, the angle between the light vector
		 and the surface normal is greater than 90 degrees and the light
		 will not contribute anything to the shade. */
		if (cos_theta < 0.0)
		{ 
			cos_theta = 0.0;
		}
		
		/* Calculate diffuse reflection components */
		diffuse_red = the_light->the_rgb.r * my_material->diffuse_rgb.r * cos_theta;
		diffuse_green = the_light->the_rgb.g * my_material->diffuse_rgb.g * cos_theta;
		diffuse_blue = the_light->the_rgb.b * my_material->diffuse_rgb.b * cos_theta;
		
		
		// Calculate cosn_alpha, the cosine to the nth power of alpha.  Alpha  
		// is the angle between the sight vector and the reflection vector.
		// n is the "glossiness" property of the material and reflects how
		// rapidly the contribution of the specular reflective component falls
		// off as alpha increases.
		//		h = unit_vector(vector_to_eye) + 
		//		  ((unit_vector(the_light.the_direction)) * (-1.0));
		
		
		//		check = unit_vector(h) * unit_vector(the_surface_normal);
		if (check < 0.0)
		  check = 0.0;
		
		tempca = cos(2.0 * (acos(check)));
		
		if (tempca < 0.0)
		{
			tempca = 0.0;
		}
			
		// Calculate tempca to the nth power
		n = my_material->glossiness;
		hold = tempca;
		for (i = 1; i <= n; i++)
		{
			tempca = hold * tempca;
		}
		
		cosn_alpha = tempca;
		
		// Calculate specular reflection components
		specular_red = the_light->the_rgb.r * my_material->specular_rgb.r * cosn_alpha;
		specular_green = the_light->the_rgb.g * my_material->specular_rgb.g * cosn_alpha;
		specular_blue = the_light->the_rgb.b * my_material->specular_rgb.b * cosn_alpha;
  
		// Update the summation components
		sum_red = sum_red + diffuse_red + specular_red;
		sum_green = sum_green + diffuse_green + specular_green;
		sum_blue = sum_blue + diffuse_blue + specular_blue;
		
		// Go on to the next light source
		light_ptr = light_ptr->next;

		// Add to ambient components the component sums
		sum_red = sum_red + (ambient_light_intensity->r * my_material->ambient_rgb.r);
		sum_green = sum_green + (ambient_light_intensity->g * my_material->ambient_rgb.g);
		sum_blue = sum_blue + (ambient_light_intensity->b * my_material->ambient_rgb.b);
		
	}
	// Return the resultant shade, clamped to 0.0..1.0
	/*	return (clamp_to_max_value(sum_red, sum_green, sum_blue));
	 */
	
}
