/*
 * LibGGI3D 
 * 
 * Z-buffer HSR rasterizer
 * 
 * Copyright (C) 1998 Jon Taylor [taylorj@ggi-project.org]
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ggi3d/modules/globals.h>
#include <ggi3d/modules/zbuffer.h>
#include <ggi3d/modules/phong_light.h>
#include <ggi3d/modules/debug.h>
#include <glib.h>
#include <ggi/ggi.h>

int coord_compare(intersection *i1, intersection *i2);
void coord_swap(intersection *i1, intersection *i2);
void render_zbuffer(GList *polygon_list, GList *light_list, ggi_visual_t *vis);

char identstr_zbuffer[] = "zbuffer 0.1";

/* We have no private data yet */
typedef struct 
{
	unsigned int junk;
} zbuffer_private;

ggi3d_modp ggi3d_dlinit()
{
	ggi3d_modp modinfo;
	zbuffer_private *priv;

	modinfo = malloc(sizeof(ggi3d_module));
	modinfo->destroy = zbuffer_destroy;
	modinfo->handle_input = zbuffer_handle;
	modinfo->command = zbuffer_command;
	
	priv = malloc(sizeof(zbuffer_private));
	modinfo->priv = priv;
	
	return modinfo;
}


void zbuffer_dump_input(zbuffer_input_t *zbuffer_input);
void zbuffer_dump_output();

/* Dump input */
void zbuffer_dump_input(zbuffer_input_t *zbuffer_input)
{
}

/* Dump output */
void zbuffer_dump_output()
{
}

/* Free the private data */
void zbuffer_destroy(ggi3d_modp self)
{
	free(self->priv);
	return;
}

int zbuffer_command(ggi3d_modp self, int cmd, void *data_in, void *data_out)
{
        ggi3d_module_id GGI3D_DEBUG;
	ggi3d_modp debug;
	void *junk;
	object debug_input;
	zbuffer_input_t *zbuffer_input = (zbuffer_input_t *)data_in;
	
	switch (cmd)
	{
		case CMD_GET_IDENT:
		data_out = (void *)identstr_zbuffer;
		return 1;
		break;
		case CMD_PROCESS:
		zbuffer_handle(self, data_in, data_out);
		return 1;
		break;
	}
	return 0;
}

int zbuffer_handle(ggi3d_modp self, void *data_in, void *data_out)
{
	zbuffer_input_t *zbuffer_input = (zbuffer_input_t *)data_in;

	render_zbuffer(zbuffer_input->polygon_list, 
		       zbuffer_input->light_list, 
		       zbuffer_input->visual);

	return 0; /* We always handle all input in one gulp */
}

int zbuffer_construct(ggi3d_modp self)
{
	zbuffer_private *priv;
	
	if (NULL==(priv=self->priv=malloc(sizeof(zbuffer_private)))) 
	  return 1;
	self->destroy		= zbuffer_destroy;
	self->handle_input	= zbuffer_handle;
	self->command		= zbuffer_command;

	return 0;
}


int coord_compare(intersection *i1, intersection *i2)
{
	return (i1->y < i2->y) || ((i1->y == i2->y) && (i1->x < i2->x));
}

void coord_swap(intersection *i1, intersection *i2)
{
	intersection temp = *i1;
	
	i1->x = i2->x;
	i1->y = i2->y;
	i2->x = temp.x;
	i2->y = temp.y;
}

void render_zbuffer(GList *polygon_list, GList *light_list, ggi_visual_t *vis)
{
	float zbuffer[640][480];
	ggi_pixel cbuffer[640][480]; // FIXME: Just black=0 and white!=0 for now
	ggi_color map;
	ggi_pixel white, black;
	ggi3d_modp mod_phong_light;
	ggi_mode mode;
	
	GList *index, *interp1, *interp2;
	
	GList *polygon_index;
	polygon *current_polygon;
	ggi_color polygon_color;
	
	GList *vertex_index;
	vertex *vertex_1;
	vertex *vertex_2;
	
	GList *intersection_list;
	GList *intersection_list_ptr;
	GList *intersection_list_start_ptr;
	GList *intersection_list_pred_ptr;
	
	intersection *inter, *prev_inter;
	
	phong_light_input_t phong_light_input;
	
	int x, y, yval, i, ii, done, qq;
	int xmax = 640;
	int ymax = 480; // Vertical resolution
	
	mod_phong_light = ggi3d_load_module(vis, "phong_light");
	
	vertex_index = g_list_alloc();
	intersection_list = g_list_alloc();
	
	ggiGetMode(vis, &mode);
	xmax = mode.virt.x;
	ymax = mode.virt.y;
	
	map.r = map.g = map.b = 0xffff;
	white = ggiMapColor(vis, &map);
	map.r = map.g = map.b = 0;
	black = ggiMapColor(vis, &map);
	
	ggiSetReadFrame(vis, 1);
	ggiSetWriteFrame(vis, 1);

	for (x = 0; x < xmax; x++)
	{
		for (y = 0; y < ymax; y++)
		{
			zbuffer[x][y] = 999.0;
			cbuffer[x][y] = black;
		}
	}
	
	// Loop through all polygons on the PVP list
	polygon_index = g_list_first(polygon_list);

	do
	{
		current_polygon = (polygon *)(polygon_index->data);
//		printf("current_polygon=(%x)\n", current_polygon);
		// Find the poly color once if it is going to be the same everywhere
		//		if (the_shading_mode = facet)
		
//		phong_light_input.light_list = light_list;
//		phong_light_input.the_material = &(current_polygon->the_material);
		
//		ggi3d_handle_input(phong_light, &phong_light_input, &polygon_color);
		
		// Rasterize the polygon by building, for each scan line, a sorted list
		// of the x intersections of pairs of polygon edges with the scan line.
		
		// First we need to determine the minimum and maximum y values of all
		// the vertices, which will determine the set of scan lines needed for
		// the scan line edge intersection list.

		vertex_index = g_list_first(current_polygon->vertex_list);
		vertex_1 = (vertex *)(vertex_index->data);
		vertex_index = vertex_index->next;
		vertex_2 = (vertex *)(vertex_index->data);
	
		// Loop through all edges of the polygon and calculate the points
		// making up each, one point per scanline.
		i = 1;
		done = 0;
		while (done != 2)
		{
			float x1, x2, y1, y2, z1, z2;
			float slope_x, slope_z;
			
			int x = x1;
			int y;
			int z = z1;
			
			if (done == 1)
			  done = 2;

			ii = 0;
			if (vertex_1->c.y != vertex_2->c.y)
			{
				if (vertex_1->c.y < vertex_2->c.y)
				{
					x1 = vertex_1->c.x;
					x2 = vertex_2->c.x;
					y1 = vertex_1->c.y;
					y2 = vertex_2->c.y;
					z1 = vertex_1->c.z;
					z2 = vertex_2->c.z;
					printf("first case\n");
					qq=1;
				}
				else
				{
					x1 = vertex_2->c.x;
					x2 = vertex_1->c.x;
					y1 = vertex_2->c.y;
					y2 = vertex_1->c.y;
					z1 = vertex_2->c.z;
					z2 = vertex_1->c.z;
					printf("second case\n");
					qq=2;
				}
		       
				slope_x = (x2 - x1) / (y2 - y1);
				printf("slope_x=%f\n",slope_x);
				slope_z = (z2 - z1) / (y2 - y1);
				
				x = x1;
				z = z1;
				
				// Loop through all scanlines that intersect the edge
				for (yval = floor(y1 + 1); yval <= floor(y2); yval++)
				{
					intersection *the_intersection_p, the_intersection;
					
					the_intersection_p = malloc(sizeof(intersection));
					
					the_intersection.x = floor(x);
					the_intersection.y = (float)yval;
					the_intersection.z = z;
					the_intersection.a = 0.0;
					
					*the_intersection_p = the_intersection;
					
					index = g_list_append(intersection_list, the_intersection_p);
					index->data = the_intersection_p;
					
					ii++;
					
					x = x + slope_x;
					z = z + slope_z;
				}
			}

			// Go on to the next edge.  If vertex_1 is the last vertex in the
			// list, vertex_2 is reset to the first one
			if (done != 2)
			{
				vertex_1 = vertex_2;
				vertex_index = vertex_index->next;
				if (vertex_index == g_list_last(vertex_index))
				{
					vertex_2 = (vertex *)(g_list_first(current_polygon->vertex_list)->data);
					done = 1;
				}
				else
				{
					vertex_2 = (vertex *)(vertex_index->data);
				}
			}
			i++;

		}

		// Sort the intersection list primarily by y-coordinate and
		// secondarily by x-coordinate
		
		interp1 = g_list_first(intersection_list)->next;
		while (interp1 != g_list_last(interp1))
		{
			interp2 = g_list_first(intersection_list)->next;
			while (interp2 != g_list_last(interp2))
			{
				inter = (intersection *)interp2->data;
				prev_inter = (intersection *)interp2->prev->data;
				
				if ((coord_compare(inter, prev_inter)))
				{
					coord_swap(prev_inter, inter);
				}
				interp2 = interp2->next;
			}
			interp1 = interp1->next;
			i++;
		}

		i = 0;
		// Render the rasterized polygon
		while (intersection_list != g_list_last(intersection_list))
		{
			intersection *coord_1 = (intersection *)intersection_list->data;
			intersection *coord_2;
			float slope, z, x1, x2, y1;
			int x, xval;
			
			intersection_list = intersection_list->next;
			coord_2 = (intersection *)intersection_list->data;
			
			if (coord_2->x != coord_1->x)
			  slope = (coord_2->z - coord_1->z) / (coord_2->x - coord_1->x);
			z = coord_1->z;
			x1 = coord_1->x;
			x2 = coord_2->x;
			y1 = coord_1->y;
			
			for (xval = (int)(x1); xval <= (int)(x2)-1; xval++)
			{
//				if (zbuffer[xval][(int)floor(y1)] < z)
//				{
					zbuffer[xval][(int)(y1)] = z;
					cbuffer[xval][(int)(y1)] = white;
//				}
				z = z + slope;
			}
			intersection_list = intersection_list->next;
		}
		// Go on to the next polygon in the list
		polygon_index = polygon_index->next;
	} while (polygon_index != g_list_last(polygon_index));
	
	i = 0;
	
	for (x = 0; x < xmax; x++)
	{
		for (y = 0; y < ymax; y++)
		{
			/* Only black&white for now */
			if (cbuffer[x][y] == white)
			{
				ggiDrawPixel(vis, x, y);
			}
			
		}
	}
	
}
