/*
******************************************************************************

   Demo of the VIDEO extension.

   Authors:	2001 Christoph Egger	[Christoph_Egger@t-online.de]
  
   This code is placed in the public domain and may be used freely for any
   purpose.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>

#include <ggi/ggi.h>
#include <ggi/video.h>


static int do_demo(ggi_visual_t vis)
{
	int rc;
	ggiVideo_t video;

	ggi_color col;
	ggi_mode *mode;
	int width, height;
	int inc, temp;
	char *pr;

	struct ggiGA_resource_props props_motor;
	struct ggiGA_resource_props props_tank;
	struct ggiGA_resource_props props_carb;

	ggiGA_resource_list reqlist = NULL;
	ggiGA_resource_list reslist = NULL;
	ggiGA_resource_handle frame_handle = NULL;
	ggiGA_resource_handle video_ptr_motor = NULL;
	ggiGA_resource_handle video_ptr_tank = NULL;
	ggiGA_resource_handle video_ptr_carb = NULL;


	ggiVideoClearMotorProperties(&props_motor);
	ggiVideoClearTankProperties(&props_tank);
	ggiVideoClearCarbProperties(&props_carb);


	printf("call ggiGAAddSimpleMode...");
	rc = ggiGAAddSimpleMode(vis, &reqlist, GGI_AUTO, GGI_AUTO,
			GGI_AUTO, GT_AUTO, &frame_handle);
	printf("done -- rc: %i\n", rc);



	props_motor.sub.motor.grid_start.x = 0;
	props_motor.sub.motor.grid_start.y = 0;
	props_tank.size.area.x = 640;
	props_tank.size.area.y = 480;
	props_tank.sub.tank.gt = GT_AUTO;

	printf("Add Video...");
	rc = ggiVideoAdd(&reqlist,
			&props_motor, &props_carb, &props_tank, NULL, 
			GA_RT_VIDEO_STREAM,
			&video_motor, &video_carb, &video_tank, NULL);
			
	printf("done -- rc: %i, video (motor/tank/carb) %p,%p,%p\n", rc,
		video_motor, video_tank, video_carb);


	printf("call ggiGAanprintf() before ggiGACheck()\n");
	ggiGAanprintf(vis, reqlist, 1024, NULL, &pr);
	printf("%s\n", pr);
	free(pr);

	rc = ggiGACheck(vis, reqlist, &reqlist);
	if (rc != GALLOC_OK) {
		printf("check failed: %i!\n", rc);
		return -1;
	}	/* if */

	printf("call ggiGAanprintf() after ggiGACheck()\n");
	ggiGAanprintf(vis, reqlist, 1024, NULL, &pr);
	printf("%s\n", pr);
	free(pr);

	rc = ggiGASet(vis, reqlist, &reslist);
	if (rc != GALLOC_OK) {
		printf("set failed: %i!\n", rc);
		return -1;
	}	/* if */

	printf("reqlist.....\n");
	ggiGAanprintf(vis, reqlist, 1024, NULL, &pr);
	printf("%s\n", pr);
	free(pr);

	printf("reslist.....\n");
	ggiGAanprintf(vis, reslist, 1024, NULL, &pr);
	printf("%s\n", pr);
	free(pr);

	frame_handle = ggiGAHandle(reslist, reqlist, frame_handle);
	video_motor = ggiGAHandle(reslist, reqlist, video_motor);
	video_carb = ggiGAHandle(reslist, reqlist, video_carb);
	video_tank = ggiGAHandle(reslist, reqlist, video_tank);


	printf("call ggiVideoTie()...\n");
	video = ggiVideoTie(vis, video_motor, video_carb, video_tank);
	if (!video) {
		printf("tie failed!\n");
		return -1;
	}	/* if */


	/* Set foreground color to white */
	col.r = 0xffff;
	col.g = 0xffff;
	col.b = 0xffff;
	ggiSetGCForeground(vis, ggiMapColor(vis, &col));

	/* Check what mode we actually got */
	mode = ggiGAGetGGIMode(frame_handle);

	width = mode->visible.x;
	height = mode->visible.y;


	printf("draw bunch of lines...\n");

	/* Draw a bunch of lines... */
	inc = height / 66;
	for (temp = 0; temp <= (height-inc); temp += inc) {
		ggiDrawLine(vis, 0, temp, width-1, height-temp);
	}	/* for */

	/* ...and flush the screen */
	ggiFlush(vis);


	printf("call ggiVideoSetImage()...\n");
	ggiVideoSetImage(video, vis, 640,480, NULL);

	printf("call ggiVideoShow()...\n");
	ggiVideoShow(video);
	for (;;) {

		static int hide = 0;
		ggi_event ev;

		struct timeval timeout={0,1000};

		ggiEventPoll(vis, emAll, &timeout);
		usleep(timeout.tv_usec);

		while (ggiEventsQueued(vis, emAll)) {
			ggiEventRead(vis, &ev, emAll);

			switch(ev.any.type) {
			case evKeyPress:
			case evKeyRepeat:

				switch(ev.key.sym) {
				case GIIK_Enter:
					hide ^= 1;

					if (hide) {
						ggiVideoHide(video);
					} else {
						ggiVideoShow(video);
					}	/* if */
					break;

				case GIIUC_Escape:
					goto cleanup;

				}	/* switch */
			}	/* switch */	
		}	/* while */


		ggiFlush(vis);
	}	/* for */

	ggiVideoHide(video);
	ggiVideoHide(video);


cleanup:


	rc = ggiVideoUntie(&video);
	printf("ret-code from ggiVideoUntie: %i\n", rc);

	rc = ggiGAReleaseList(vis, &reqlist);
	printf("ret-val from ggiGAReleaseList: %i\n", rc);
	printf("reqlist.....\n");
	ggiGAanprintf(vis, reqlist, 1024, NULL, &pr);
	printf("%s", pr);
	free(pr);

	printf("Final current resource list.....\n");
	ggiGAEmptyList(&reqlist);
	ggiGAGet(vis, &reqlist);
	ggiGAanprintf(vis, reqlist, 1024, NULL, &pr);
	printf("%s", pr);
	free(pr);

	ggiGAEmptyList(&reqlist);

	return 0;
}	/* do_demo */



int main(void)
{
	ggi_visual_t vis;

	/* Initialize LibGGI */
	if (ggiInit() != 0) {
		fprintf(stderr, "Unable to initialize LibGGI\n");
		exit(1);
	}	/* if */
	
	/* Initialize VIDEO extension */
	if (ggiVideoInit() != 0) {
		ggiPanic("Unable to initialize LibVIDEO extension\n");
	}	/* if */

	/* Open the default visual */
	if ((vis = ggiOpen(NULL)) == NULL) {
		ggiPanic("Unable to open default visual\n");
	}	/* if */
	
	/* Turn on asynchronous mode (which should always be used) */
	ggiSetFlags(vis, GGIFLAG_ASYNC);
	

	/* Attach the VIDEO extension to the visual */
	if (ggiVideoAttach(vis, NULL) < 0) {
		ggiPanic("Unable to attach VIDEO extension to visual\n");
	}	/* if */


	/* Try the API functions */
	do_demo(vis);

	
	/* Detach extension from visual */
	ggiVideoDetach(vis);

	/* Close visual */
	ggiClose(vis);

	/* Deinitialize VIDEO extension */
	ggiVideoExit();

	/* Deinitialize LibGGI */
	ggiExit();

	return 0;
}	/* main */
