/*
******************************************************************************

   LibVIDEO: extension API header file

   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


******************************************************************************
*/

#ifndef _GGI_VIDEO_H
#define _GGI_VIDEO_H

#ifndef _GGI_INTERNAL_GII_H
#include <ggi/ggi.h>
#endif

__BEGIN_DECLS

#include <ggi/galloc.h>

#include <ggi/buf.h>
#include <ggi/ovl.h>
#include <ggi/blt.h>
#include <ggi/gpf.h>


/* Extension management
 */
int ggiVideoInit(void);
int ggiVideoExit(void);

int ggiVideoAttach(ggi_visual_t vis, ggiGA_resource_list *reqlist);
int ggiVideoDetach(ggi_visual_t vis);


/* Structure
 */

#ifdef HAVE_STRUCT_VIDEO
typedef struct ggiVideo *ggiVideo_t;
#else
struct ggiVideo { };

typedef struct ggiVideo *ggiVideo_t;
#endif


/* Flags
 */

enum ggiVideo_flags {
	/* priority flag */
	VIDEO_FLAG_ALWAYS_ON_TOP	= OVL_FLAG_ALWAYS_ON_TOP,

	/* video format */
	VIDEO_FLAG_FORMAT_MPEG_1	= OVL_FLAG_FORMAT_MPEG_1,
	VIDEO_FLAG_FORMAT_MPEG_2	= OVL_FLAG_FORMAT_MPEG_2,
	VIDEO_FLAG_FORMAT_MPEG_4	= OVL_FLAG_FORMAT_MPEG_4,
	VIDEO_FLAG_FORMAT_AVI		= OVL_FLAG_FORMAT_AVI,
	VIDEO_FLAG_FORMAT_QUICKTIME	= OVL_FLAG_FORMAT_QUICKTIME,
	VIDEO_FLAG_FORMAT_H261		= OVL_FLAG_FORMAT_H261,
	VIDEO_FLAG_FORMAT_H263		= OVL_FLAG_FORMAT_H263,

	/* operations during display */
	VIDEO_FLAG_INTERPOLATE		= OVL_FLAG_INTERPOLATE,

	/* framerate flags */
	VIDEO_FLAG_CURRENT_FRAMERATE	= OVL_FLAG_CURRENT_FRAMERATE,
	VIDEO_FLAG_AVERAGE_FRAMERATE	= OVL_FLAG_AVERAGE_FRAMERATE,

};	/* ggiVideo_flags */



/* API functions
 */

/* type is always a subtype of GA_RT_VIDEO_*.
 * Otherwise, everything fails.
 */


ggiVideo_t ggiVideoCreateSimpleVideo(ggi_visual_t vis,
			long width, long height,
			enum ggiGA_resource_type type,
			enum ggiVideo_flags flags);

int ggiVideoClearMotorProperties (struct ggiGA_resource_props *prop);
int ggiVideoClearCarbProperties  (struct ggiGA_resource_props *prop);
int ggiVideoClearTankProperties  (struct ggiGA_resource_props *prop);

ggiVideo_t ggiVideoCreate(ggi_visual_t vis,
		      struct ggiGA_resource_props *prop_motor,
		      struct ggiGA_resource_props *prop_carb,
		      struct ggiGA_resource_props *prop_tank,
		      enum ggiGA_resource_type type);

ggiVideo_t ggiVideoCreateInGroup(ggi_visual_t vis, ggiVideo_t group_leader,
			 	struct ggiGA_resource_props *prop_tank);

int ggiVideoDestroy(ggiVideo_t *video);


ggiGA_resource_handle ggiVideoGetMotorHandle(ggiVideo_t video);
ggiGA_resource_handle ggiVideoGetCarbHandle(ggiVideo_t video);
ggiGA_resource_handle ggiVideoGetTankHandle(ggiVideo_t video);

gpf_handle_t ggiVideoGetGPFHandle(ggiVideo_t video);


#define ggiVideoAddFlags(video, flags)		\
	ggiVideoSetFlags(video, ggiVideoGetFlags(video) | flags);

#define ggiVideoRemoveFlags(video, flags)	\
	ggiVideoSetFlags(video, ggiVideoGetFlags(video) & ~flags);

int ggiVideoSetFlags(ggiVideo_t video, enum ggiVideo_flags flags);
enum ggiVideo_flags ggiVideoGetFlags(ggiVideo_t video);




/* Get out the video data from a certain frame
 * a ggiBuf_t. Counting starts from 0, which is _always_
 * the video data from the fram currently being displayed.
 */
ggiBuf_t ggiVideo2Buf(ggiVideo_t video, int num_frame);


/* Set the cache-size to be able to cache "num_frames".
 * Negative return-value indicates an error, 0 means success,
 * > 0 means, not enough memory to cache "num_frames".
 * Exampe: You wanna cache 10 frames and it returns 5, then
 * it means, there were only enough memory to cache 5 frames.
 */
int ggiVideoSetCache(ggiVideo_t video, int num_frames);


/* When you have several overlapping videos, you can specify
 * which one should be in the front by setting a priority.
 * The higher the priority, the more in the front it is.
 * The behaviour is compareable with the layer management of
 * an professional image manipulation program.
 *
 * TODO: We have to think about how to interfere with overlay resources,
 *	 which may even have a higher priority (i.e. sprite-pointer).
 */
int ggiVideoSetPriority(ggiVideo_t video, uint32 slot, int swap);
int ggiVideoGetPriority(ggiVideo_t video, uint32 *slot);



/* Note, this don't change the internal coordbase. It merely tells
 * the video resource, which coordbase the params of the following
 * called APIs (i.e. ggiVideoSetPos(), ggiVideoMove(), ...) have.
 * This allows to set some values in dots even the video resource
 * only uses pixels internally.
 */
int ggiVideoSetCoordBase(ggiVideo_t video, unsigned int val);


/* Set the given image (vis) to the video-area with grabbing the image
 * from a visual at pos (x,y)
 */
int ggiVideoSetImage(ggiVideo_t video, ggi_visual_t vis,
			long x, long y, unsigned char *alpha);


/* Get some useful information.
 */
int ggiVideoGetFrameRate(ggiVideo_t video, enum ggiVideo_flags flags);
int ggiVideoGetVideoProperties(ggiVideo_t video, struct gpf_fileinfo_t **info);


/* set the video-area to the position (x,y), where (x,y) is the top-left
 * corner of the video-area. If the video must be positioned on a
 * "snap to" grid, then the actual coordinates used will be rounded
 * down to nearest coordinates that fit on the grid.
 */
int ggiVideoSetPos(ggiVideo_t video, long x, long y);
int ggiVideoGetPos(ggiVideo_t video, long *x, long *y);


/* moves the video at (x,y) If the video must be positioned
 * on a "snap to" grid, then the actual coordinates used will be
 * rounded down to nearest coordinates that fit on the grid.
 */
int ggiVideoMove(ggiVideo_t video, long x, long y);





/* set the size of the video-area. Note, this doesn't change the
 * storage size. It works for resources, that can do pixel doubling,
 * for example, otherwise it fails. If the video must be positioned
 * on a "snap to" grid, then the given size (w,h) will be rounded down
 * to nearest coordinates that fit on the grid.
 */
int ggiVideoSetSize(ggiVideo_t video, long w, long h);
int ggiVideoGetSize(ggiVideo_t video, long *w, long *h);


int ggiVideoSetZoom(ggiVideo_t video, long x, long y, long w, long h, int keep_ratio);
int ggiVideoGetZoom(ggiVideo_t video, long *x, long *y, long *w, long *h);



/* shows/hides the video-area
 */
int ggiVideoShow(ggiVideo_t video);
int ggiVideoHide(ggiVideo_t video);
int ggiVideoIsHidden(ggiVideo_t video);

int ggiVideoIsVideoEmulated(ggiVideo_t video);


/* Hiding and showing of _emulated_ Videos.
 * Has no effect, if true hw-accelerated.
 * It is intended to be used when the background
 * changes to avoid flickering.
 */
int ggiVideoHideIfEmulated(ggiVideo_t video);
int ggiVideoShowIfEmulated(ggiVideo_t video);



/* enables/disables the video-area
 */
int ggiVideoEnable(ggiVideo_t video);
int ggiVideoDisable(ggiVideo_t video);
int ggiVideoIsEnabled(ggiVideo_t video);



/* Video Control
 */
int ggiVideoOpen(ggiVideo_t video, const char *source, char *srcargf, ...);
int ggiVideoClose(ggiVideo_t video);

int ggiVideoStart(ggiVideo_t video);
int ggiVideoStop(ggiVideo_t video);

int ggiVideoPause(ggiVideo_t video);
int ggiVideoContinue(ggiVideo_t video);

int ggiVideoIsPlaying(ggiVideo_t video, gpfIndicatorCallback cbI);


int ggiVideoForward(ggiVideo_t video, int num_frames);
int ggiVideoBackward(ggiVideo_t video, int num_frames);
int ggiVideoRewind(ggiVideo_t video, int restart);
int ggiVideoSetFramePos(ggiVideo_t video, int num_frame);
int ggiVideoGetFramePos(ggiVideo_t video, int *num_frame);


/**********************************************
 *	Sound
 */


enum ggiVideoSound_flags {
	VIDEO_FLAG_SOUND_START		= 0x00000001,
	VIDEO_FLAG_SOUND_STOP		= 0x00000002,
	VIDEO_FLAG_SOUND_PAUSE		= 0x00000004,
	VIDEO_FLAG_SOUND_CONTINUE	= 0x00000008,
	VIDEO_FLAG_SOUND_FETCH		= 0x00000010,
};



/* Get the sound properties (sampling-rate, mono/stereo, 8bpp/16bpp, etc.)
 * This data is a void * to be able to pass every soundinformation to the application.
 * As the application know, which video-format it uses, it also knows, which soundformat
 * the video has. This let us play both wave and midi sound.
 */
int ggiVideoGetSoundProperties(ggiVideo_t video, void *data, size_t *data_size);


/* Callbacks:
 *
 * These must be implemented by the applications themself. They may make use of
 * any soundlib. The callbacks are called internally automatically.
 */

typedef int (*ggiVideoSoundStart)(void);
typedef int (*ggiVideoSoundStop)(void);

typedef int (*ggiVideoSoundPause)(void);
typedef int (*ggiVideoSoundContinue)(void);

/* Forward, backward, rewind, etc. is not needed here, because
 * you always get appropriate sound-data.
 * Negative return value means an error occured.
 * > 0 means, there is still data to fetch.
 * = 0 means, no more data to fetch.
 */
typedef int (*ggiVideoSoundFetchData)(void *data, size_t data_size);


/* If you don't set all callbacks, then don't wonder, when
 * soundplaying doesn't work as expected!
 * If you don't use it at all, you'll never hear any sound, because
 * all sound-data will be dropped.
 */
int ggiVideoSetCallbacks(ggiVideo_t video, enum ggiVideoSound_flags cb_flag, void *callback);



__END_DECLS

#endif /* _GGI_VIDEO_H */
