/*
******************************************************************************

   LibVIDEO: (de)alloction functions

   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/video.h>
#include <ggi/internal/video_debug.h>
#include <ggi/internal/galloc_ops.h>


int dummyfunc(void *dummy)
{
	return -1;
}	/* dummyfunc */


ggiVideo_t ggiVideoCreate(ggi_visual_t vis,
		      struct ggiGA_resource_props *prop_motor,
		      struct ggiGA_resource_props *prop_carb,
		      struct ggiGA_resource_props *prop_tank,
		      enum ggiGA_resource_type type)
{
	int rc;

	ggiVideo_t video;
	ggiOvl_t ovl;
	ggiBlt_t blt;
	ggiBuf_t buf;

	video = calloc(1, sizeof(*video));
	if (!video) {
		/* not enough memory */
		goto err0;
	}	/* if */


	/* Make copies of the props, so that we can modify them later in the targets
 	 * to be able to retry to use libovl again and again, in case it fails below.
	 */
	memcpy(&video->prop_motor, prop_motor,
		sizeof(struct ggiGA_resource_props));
	memcpy(&video->prop_carb, prop_carb,
		sizeof(struct ggiGA_resource_props));
	memcpy(&video->prop_tank, prop_tank,
		sizeof(struct ggiGA_resource_props));


	/* create buffer containing the content being displayed */
	buf = ggiBufCreateBuffer(vis, );
	if (buf == NULL) {
		/* failed */
		goto err0;
	}	/* if */

	video->buf = buf;

	/* create second buffer (same prameters as above) being used
	 * when disabled
	 */
	buf = ggiBufCreateBuffer(vis, );
	if (buf == NULL) {
		/* failed */
		goto err1;
	}	/* if */
	
	video->buf_disabled = buf;


	video->vis = vis;
	video->fileinfo = NULL;


	/* We position the video at (0,0), which is the top-left-corner
	 * of the screen.
	 */
	video->x = video->cur_x = 0;
	video->y = video->cur_y = 0;

	LIB_ASSERT(video->x != GGI_AUTO);
	LIB_ASSERT(video->y != GGI_AUTO);

	video->w = video->cur_w = video->prop_tank.size.area.x;
	video->h = video->cur_h = video->prop_tank.size.area.y;

	video->hidecount = 1;           /* Come up hidden! */
	video->disablecount = 0;        /* Come up enabled (active) */
	video->zoomed = 0;              /* not zoomed */
	video->slot = 0;                /* default value */

	video->cb = ggiGAGetCoordBase(video->res_handle_tank);

	/* sound */
	video->soundstart = (void *)dummyfunc;
	video->soundstop = (void *)dummyfunc;
	video->soundpause = (void *)dummyfunc;
	video->soundcontinue = (void *)dummyfunc;
	video->soundfetchdata = (void *)dummyfunc;


	video->vtype = VIDEO_VIDEO;

	ovl = ggiOvlCreate(vis, prop_motor, prop_carb, prop_tank, type);
	if (ovl != NULL) {
		/* success */
		video->subtype.video = ovl;
		return rc;
	}	/* if */


	video->vtype = VIDEO_EMUVIDEO;

	memset(video->subtype.emuvideo, 0, sizeof(struct ggi_emuvideo_t));


	/* Nothing more to do here.
	 * The emuvideo stuff is allocated inside of the target.
	 * Because targets tend to be (re)loaded several times, make sure,
	 * that this will be done only once to avoid memory leaks.
	 */	

	return video;

 err1:
	rc = ggiBufDestroy(video->buf);
	LIB_ASSERT(rc == GALLOC_OK);
 err0:
	return NULL;
}	/* ggiVideoCreate */



ggiVideo_t ggiVideoCreateSimpleVideo(ggi_visual_t vis,
			   long width, long height,
			   enum ggiGA_resource_type subtype,
			   enum ggiVideo_flags flags)
{
	struct ggiGA_resource_props props_motor;
	struct ggiGA_resource_props props_carb;
	struct ggiGA_resource_props props_tank;


	ggiVideoClearMotorProperties(&props_motor);
	ggiVideoClearCarbProperties(&props_carb);
	ggiVideoClearTankProperties(&props_tank);

	props_tank.flags = flags;
	props_tank.size.area.x = width;
	props_tank.size.area.y = height;

	return ggiVideoCreate(vis, &props_motor, &props_carb,
	                    &props_tank, subtype);
}	/* ggiVideoCreateVideo */



int ggiVideoDestroy(ggiVideo_t *video)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;
	if (video[0] == NULL) return GGI_EARGINVAL;


	switch (video[0]->vtype) {
	case VIDEO_VIDEO:
		rc = ggiOvlDestroy(&(video[0]->subtype.emuvideo.ovl));
		LIB_ASSERT(rc == GALLOC_OK);
		break;

	case VIDEO_EMUVIDEO:
		if (video[0]->subtype.emuvideo.ovl != NULL) {
			/* partly emulated */
			rc = ggiOvlDestroy(&(video[0]->subtype.emuvideo.ovl));
			LIB_ASSERT(rc == GALLOC_OK);
			break;
		}	/* if */

		/* full emulated */
		rc = ggiBltDestroy(&(video[0]->subtype.emuvideo.bob));
		LIB_ASSERT(rc == GALLOC_OK);
		free(video[0]->subtype.emuvideo.backgr);
		break;

	}	/* switch */

	free(video[0]);
	video[0] = NULL;

	return rc;
}	/* ggiVideoDestroy */
