/*
******************************************************************************

   LibVIDEO: control functions

   Copyright (C) 2001 Christoph Egger   [Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <ggi/internal/video.h>
#include <ggi/internal/video_debug.h>
#include <ggi/internal/galloc_ops.h>

#include <ggi/gpf.h>
#include <stdio.h>
#include <stdarg.h>


int ggiVideoOpen(ggiVideo_t video, const char *source, char *srcargf, ...)
{
	int rc = GALLOC_OK;
	void *va_srcargptr;
	va_list src_args;


	va_start(src_args, source);

	va_argptr = va_arg(src_args, void *);
	va_end(src_args);


	/* open data stream, input = source and output = libbuf */
	video->gpf_handle = gpfOpen(source, srcargf, "libbuf", "(%p)", va_srcargptr, video->buf);
	if (!video->gpf_handle) {
		/* failed */
		rc = gpfGetOpenError();
		goto exit;
	}	/* if */

	rc = gpfGetFileInfo(video->gpf_handle, &(video->fileinfo));
	LIB_ASSERT(rc == GALLOC_OK);

exit:
	return rc;
}	/* ggiVideoOpen */


int ggiVideoClose(ggiVideo_t video)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfClose(video->gpf_handle);
	LIB_ASSERT(rc == GALLOC_OK);

	return rc;
}	/* ggiVideoClose */


int ggiVideoStart(ggiVideo_t video)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	/* start data transfer */
	rc = gpfStartStream(video->gpf_handle);
	LIB_ASSERT(rc == GALLOC_OK);

	video->soundstart();

	return rc;
}	/* ggiVideoStart */


int ggiVideoStop(ggiVideo_t video)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfStopStream(video->gpf_handle);
	LIB_ASSERT(rc == GALLOC_OK);

	video->soundstop();

	return rc;
}	/* ggiVideoStop */


int ggiVideoPause(ggiVideo_t video)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfStopStream(video->gpf_handle);
	LIB_ASSERT(rc == GALLOC_OK);

	video->soundpause();

	return rc;
}	/* ggiVideoPause */


int ggiVideoContinue(ggiVideo_t video)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfStartStream(video->gpf_handle);
	LIB_ASSERT(rc == GALLOC_OK);

	video->soundcontinue();

	return rc;
}	/* ggiVideoContinue */


int ggiVideoIsPlaying(ggiVideo_t video, gpfIndicatorCallback cbI)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfDoNextStreamDataBlock(video->gpf_handle, cbI);
	LIB_ASSERT(rc == GALLOC_OK);

	video->soundfetchdata(video->buf, ggiBufGetSizeInBytes(video->buf));

	return (video->isplaying != 0) ? 1 : 0;
}	/* ggiVideoIsPlaying */


int ggiVideoForward(ggiVideo_t video, int num_frames)
{
	int rc;
	int cur_frame;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfGetSelectedImage(video->gpf_handle, &cur_frame);
	LIB_ASSERT(rc == GALLOC_OK);

	rc = gpfSelectImage(video->gpf_handle, cur_frame + num_frames);
	LIB_ASSERT(rc == GALLOC_OK);
	
	return rc;
}	/* ggiVideoForward */


int ggiVideoBackward(ggiVideo_t video, int num_frames)
{
	int rc;
	int cur_frame;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfGetSelectedImage(video->gpf_handle, &cur_frame);
	LIB_ASSERT(rc == GALLOC_OK);

	rc = gpfSelectImage(video->gpf_handle, cur_frame - num_frames);
	LIB_ASSERT(rc == GALLOC_OK);
	
	return rc;
}	/* ggiVideoBackward */


int ggiVideoRewind(ggiVideo_t video, int restart)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	rc = ggiVideoStop(video);
	LIB_ASSERT(rc == GALLOC_OK);

	/* Select the first image */
	rc = gpfSelectImage(video->gpf_handle, GPF_FIRST_IMAGE);
	LIB_ASSERT(rc == GALLOC_OK);

	if (restart) {
		rc = ggiVideoStart(video);
		LIB_ASSERT(rc == GALLOC_OK);
	}	/* if */
	
	return rc;
}	/* ggiVideoRewind */


int ggiVideoSetFramePos(ggiVideo_t video, int num_frame)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;

	rc = gpfSelectImage(video->gpf_handle, num_frame);
	LIB_ASSERT(rc == GALLOC_OK);

	return rc;
}	/* ggiVideoSetFramePos */


int ggiVideoGetFramePos(ggiVideo_t video, int *num_frame)
{
	int rc;

	if (video == NULL) return GGI_EARGINVAL;
	if (num_frame == NULL) return GGI_EARGINVAL;

	rc = gpfGetSelectedImage(video->gpf_handle, num_frame);
	LIB_ASSERT(rc == GALLOC_OK);

	return rc;
}	/* ggiVideoGetFramePos */



int ggiVideoSetCallbacks(ggiVideo_t video, enum ggiVideoSound_flags cb_flag, void *callback)
{
	if (video == NULL) return GGI_EARGINVAL;
	if (callback == NULL) return GGI_EARGINVAL;

	switch (cb_flag) {
	case VIDEO_FLAG_SOUND_START:
		video->soundstart = (ggiVideoSoundStart)callback;
		break;

	case VIDEO_FLAG_SOUND_STOP:
		video->soundstop = (ggiVideoSoundStop)callback;
		break;

	case VIDEO_FLAG_SOUND_PAUSE:
		video->soundpause = (ggiVideoSoundPause)callback;
		break;

	case VIDEO_FLAG_SOUND_CONTINUE:
		video->soundcontinue = (ggiVideoSoundContinue)callback;
		break;

	case VIDEO_FLAG_SOUND_FETCH:
		video->soundfetchdata = (ggiVideoSoundFetchData)callback;
		break;

	}	/* switch */

	return GALLOC_OK;
}	/* ggiVideoSetCallbacks */
