/*
******************************************************************************

   LibVIDEO: extension initialization.

   Copyright (C) 2001 Christoph Egger	[Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <string.h>

#include "config.h"
#include <ggi/internal/video.h>
#include <ggi/internal/video_debug.h>

#include <ggi/gg.h> /* We use LibGG to manage config files */

#include <ggi/galloc.h>
#include <ggi/buf.h>
#include <ggi/ovl.h>
#include <ggi/blt.h>


/* Static variables 
 */
static int 	 _videoLibIsUp       = 0;
static void	*_videoConfigHandle  = NULL;
static char      _videoconfstub[512] = VIDEOCONFFILE;
static char     *_videoconffile      = _videoconfstub+VIDEOTAGLEN;

static ggiGA_resource_list reqdummy = NULL;


int _videoDebugSync = 0;
uint32 _videoDebugState = 0;



/* Extension ID. Defaulting to -1 should make segfault on abuse more likely...
 * This is exported so that sublibs can use it.
 */
EXPORTVAR ggi_extid _videoID = -1;

#define SUBLIB_PREFIX	"VIDEOdl_"


/* Dummy function which returns -1
   We use this to reset the function pointers */
static int dummyfunc(void)
{
	VIDEODPRINT_CORE("dummyfunc() of LibVIDEO called\n");

	return GALLOC_EUNAVAILABLE;
}	/* dummyfunc */


/* Reset all function pointers to dummyfunc */
static void clearfuncs(videopriv *priv)
{
	priv->tie = (void *)dummyfunc;

	priv->setimage = (void *)dummyfunc;

	priv->setpos = (void *)dummyfunc;
	priv->move = (void *)dummyfunc;

	priv->setsize = (void *)dummyfunc;
	priv->setpriority = (void *)dummyfunc;
	priv->setflags = (void *)dummyfunc;

	priv->setzoom = (void *)dummyfunc;

	priv->getframerate = (void *)dummyfunc;

	priv->show = (void *)dummyfunc;
	priv->hide = (void *)dummyfunc;

	priv->enable = (void *)dummyfunc;
	priv->disable = (void *)dummyfunc;
}	/* clearfuncs */


/* This function is called by LibGGI when the API-list changes */
static int changed(ggi_visual_t vis, int whatchanged)
{
	VIDEODPRINT_LIBS("changed(%p, 0x%x) of LibVIDEO called\n",
			vis, whatchanged);

	switch (whatchanged) {
	case GGI_CHG_APILIST:
		{
		int i;
		const char *fname;
		ggi_dlhandle *lib;
		char api[GGI_MAX_APILEN], args[GGI_MAX_APILEN];

		clearfuncs(LIBGGI_VIDEOEXT(vis));

		for (i=0; ggiGetAPI(vis, i, api, args) == 0; i++) {
			strcat(api,"-video");
			fname = ggMatchConfig(_videoConfigHandle, api, NULL);
			if (fname == NULL) {
				/* No special implementation for this sublib */
				continue;
			}	/* if */
			lib = ggiExtensionLoadDL(vis, fname, args, NULL,
						 SUBLIB_PREFIX);
		}	/* for */
		}
		break;
	}	/* switch */
                                
	return 0;
}	/* changed */


/* Initialize the extension
 */
int ggiVideoInit(void)
{
	int err;
	char *str;

	_videoLibIsUp++;
	if (_videoLibIsUp > 1) return 0; /* Initialize only at first call. */

	str = getenv("VIDEO_DEBUG");
	if (str != NULL) {
		_videoDebugState = atoi(str);
		VIDEODPRINT_CORE("Debugging=%d\n", _videoDebugState);
	}	/* if */

	str = getenv("VIDEO_DEBUGSYNC");
	if (str != NULL) {
		_videoDebugSync = 1;
	}	/* if */


	err = ggLoadConfig(_videoconffile, &_videoConfigHandle);
	if (err != GGI_OK) {
		fprintf(stderr,"LibVIDEO: couldn't open %s\n",
			_videoconffile);
		_videoLibIsUp--;
		return err;
	}	/* if */

	_videoID = ggiExtensionRegister("VIDEO", sizeof(videopriv), changed);

	if (_videoID < 0) {
		fprintf(stderr,
			"LibVIDEO: failed to register as extension.\n");
		_videoLibIsUp--;
		ggFreeConfig(_videoConfigHandle);
		return _videoID;
	}	/* if */


	if ((err = ggiGAInit()) != 0) {
		fprintf(stderr,"LibVIDEO: couldn't Initilize LibGALLOC.\n");
		_videoLibIsUp--;
		return err;
	}	/* if */

	if ((err = ggiBufInit()) != 0) {
		fprintf(stderr,"LibVIDEO: couldn't Initilize LibBUF.\n");
		_videoLibIsUp--;
		return err;
	}	/* if */

	if ((err = ggiOvlInit()) != 0) {
		fprintf(stderr,"LibVIDEO: couldn't Initilize LibOVL.\n");
		_videoLibIsUp--;
		return err;
	}	/* if */

	if ((err = ggiBltInit()) != 0) {
		fprintf(stderr,"LibVIDEO: couldn't Initilize LibBLT.\n");
		_videoLibIsUp--;
		return err;
	}	/* if */


	if ((err = gpfInit()) != 0) {
		fprintf(stderr,"LibVIDEO: couldn't Initilize LibGPF.\n");
		_videoLibIsUp--;
		return err;
	}	/* if */


	return 0;
}	/* ggiVideoInit */


/* Deinitialize the extension
 */
int ggiVideoExit(void)
{
	int rc;
    
	if (!_videoLibIsUp) return -1;

	if (_videoLibIsUp > 1) {
		/* Exit only at last call */
		_videoLibIsUp--;
		return 0;
	}	/* if */
	
	rc = ggiExtensionUnregister(_videoID);
	ggFreeConfig(_videoConfigHandle);

	_videoLibIsUp = 0;

#warning  TODO: Check the return-code here
	rc = gpfExit();
        rc = ggiBltExit();
        rc = ggiOvlExit();
	rc = ggiBufExit();
	rc = ggiGAExit();

	return rc;
}	/* ggiVideoExit */


/* Attach the extension to a visual
 */
int ggiVideoAttach(ggi_visual_t vis, ggiGA_resource_list *reqlist)
{
	int rc;

	VIDEODPRINT_CORE("videoAttach(%p) called\n", vis);

	rc = ggiExtensionAttach(vis, _videoID);
	if (rc < 0) {
		/* error occured */
		goto err0;
	}	/* if */

	if (rc == 0) {
		rc = ggiGAAttach(vis);
		if (rc < 0) {
			goto err1;
		}	/* if */

		rc = ggiBufAttach(vis, reqlist);
		if (rc < 0) {
			goto err2;
		}	/* if */

		rc = ggiOvlAttach(vis, reqlist);
		if (rc < 0) {
			goto err3;
		}	/* if */

		rc = ggiBltAttach(vis, reqlist);
		if (rc < 0) {
			goto err4;
		}	/* if */

		rc = 0;

		/* We are actually creating the primary instance. */
		memset(LIBGGI_VIDEOEXT(vis), 0, sizeof(videopriv));
		LIBGGI_VIDEOEXT(vis)->priv = NULL;
		if (reqlist == NULL) {
			LIBGGI_VIDEOEXT(vis)->reqlist = &reqdummy;
		} else {
			LIBGGI_VIDEOEXT(vis)->reqlist = reqlist;
		}	/* if */

		/* Now fake an "API change" so the right libs get loaded */
		changed(vis, GGI_CHG_APILIST);
	}	/* if */

	return rc;

err4:
	rc = ggiOvlDetach(vis);
	LIB_ASSERT(rc == GALLOC_OK);
err3:
	rc = ggiBufDetach(vis);
	LIB_ASSERT(rc == GALLOC_OK);
err2:
	rc = ggiGADetach(vis);
	LIB_ASSERT(rc == GALLOC_OK);
err1:
	ggiExtensionDetach(vis, _videoID);
err0:
	return rc;
}	/* ggiVideoAttach */


/* Detach the extension from a visual
 */
int ggiVideoDetach(ggi_visual_t vis)
{
	int rc;

	ggiGA_resource_list list = NULL;
	ggiGA_resource_handle current = NULL;
	ggiGA_resource_handle next = NULL;

	int type;

	void *priv = NULL;


	priv = LIBGGI_VIDEOEXT(vis)->priv;

	VIDEODPRINT_CORE("videoDetach(%p) called\n", vis);

	rc = ggiExtensionDetach(vis, _videoID);
	if (rc != 0) return rc;

	rc = ggiGAGet(vis, &list);
	LIB_ASSERT(rc == GALLOC_OK);

	current = list;
	while (current != NULL) {
		if (ggiGAGetVisual(current) != vis) {
			current = ggiGAGetNext(current);
			continue;
		}	/* if */


		next = ggiGAGetNext(current);

		/* Release all sprites/videos/windows here */
		type = (ggiGAGetType(current) & GA_RT_TYPE_MASK);
		if (type == GA_RT_VIDEO) {
			rc = ggiGARelease(vis, &list, &current);
			LIB_ASSERT(rc == GALLOC_OK);
		}	/* if */

		current = next;
	}	/* while */

	rc = ggiGAEmptyList(&list);
	LIB_ASSERT(rc == GALLOC_OK);

	if (priv != NULL) {
		free(priv);
	}	/* if */


	rc = ggiBltDetach(vis);
	LIB_ASSERT(rc == GALLOC_OK);

	rc = ggiOvlDetach(vis);
	LIB_ASSERT(rc == GALLOC_OK);

	rc = ggiBufDetach(vis);
	LIB_ASSERT(rc == GALLOC_OK);

	rc = ggiGADetach(vis);
	LIB_ASSERT(rc == GALLOC_OK);

	return rc;
}	/* ggiVideoDetach */

/* This should be included by exactly _one_ file in each extension library. */
#include <ggi/internal/ggilibinit.h>
