#include <limits.h>
#include "stubsxmi.h"
//#include "mi_spans.h"
#include "mi_api.h"
#include "mi_scanfill.h"
#include "mi_ply.h"
#include "xmi_debug.h"

/*
 *     Written by Brian Kelleher;  Oct. 1985
 *
 *     This module contains all of the utility functions
 *     needed to scan-convert a polygon.  The driver subroutine,
 *     miFillGeneralPoly(), is in mi_plygen.c.
 */


/*
 *     InsertEdgeInET
 *
 *     Insert the given edge into the edge table.
 *     First we must find the correct bucket in the
 *     Edge table, then find the right slot in the
 *     bucket.  Finally, we can insert it.
 *
 */
static void XMI_stubs_insertedgeinet(ggi_visual_t vis, EdgeTable *ET, EdgeTableEntry *ETE, int scanline, ScanLineListBlock **SLLBlock, int *iSLLBlock) {
	EdgeTableEntry *start, *prev;
	ScanLineList *pSLL, *pPrevSLL;
	ScanLineListBlock *tmpSLLBlock;

	XMIDPRINT_CORE("XMI_stubs_insertedgeinet()\n");
	
	/* Find the right bucket to put the edge into */
	pPrevSLL = &ET->scanlines;
	pSLL = pPrevSLL->next;
	while (pSLL && (pSLL->scanline < scanline)) {
		pPrevSLL = pSLL;
		pSLL = pSLL->next;
	}
	
	/* Reassign pSLL (pointer to ScanLineList) if necessary */
	if ((!pSLL) || (pSLL->scanline > scanline)) {
		if (*iSLLBlock > SLLSPERBLOCK-1) {
			tmpSLLBlock =  (ScanLineListBlock *)malloc(sizeof(ScanLineListBlock));
			(*SLLBlock)->next = tmpSLLBlock;
			tmpSLLBlock->next = (ScanLineListBlock *)NULL;
			*SLLBlock = tmpSLLBlock;
			*iSLLBlock = 0;
		}
		pSLL = &((*SLLBlock)->SLLs[(*iSLLBlock)++]);
		
		pSLL->next = pPrevSLL->next;
		pSLL->edgelist = (EdgeTableEntry *)NULL;
		pPrevSLL->next = pSLL;
	}
	pSLL->scanline = scanline;

	/* Now insert the edge in the right bucket */
	prev = (EdgeTableEntry *)NULL;
	start = pSLL->edgelist;
	while (start && (start->bres.minor_axis < ETE->bres.minor_axis)) {
		prev = start;
		start = start->next;
	}
	ETE->next = start;

	if (prev)
	  prev->next = ETE;
	else
	  pSLL->edgelist = ETE;
}

/*
 *     CreateEdgeTable
 *
 *     This routine creates the edge table for
 *     scan converting polygons. 
 *     The Edge Table (ET) looks like:
 *
 *    EdgeTable
 *     --------
 *    |  ymax  |        ScanLineLists
 *    |scanline|-->------------>-------------->...
 *     --------   |scanline|   |scanline|
 *                |edgelist|   |edgelist|
 *                ---------    ---------
 *                    |             |
 *                    |             |
 *                    V             V
 *              list of ETEs   list of ETEs
 *
 *     where ETE is an EdgeTableEntry data structure,
 *     and there is one ScanLineList per scanline at
 *     which an edge is initially entered.
 *
 */

void XMI_stubs_createetandaet(ggi_visual_t vis, int count, const miPoint *pts, EdgeTable *ET, EdgeTableEntry *AET, EdgeTableEntry *pETEs, ScanLineListBlock *pSLLBlock) {
	const miPoint *top, *bottom;
	const miPoint *PrevPt, *CurrPt;
	int iSLLBlock = 0;
	int dy;

	XMIDPRINT_CORE("XMI_stubs_createetandaet()\n");
	
	if (count < 2)  
	  return;

	/* Initialize the Active Edge Table */
	AET->next = (EdgeTableEntry *)NULL;
	AET->back = (EdgeTableEntry *)NULL;
	AET->nextWETE = (EdgeTableEntry *)NULL;
	//    AET->bres.minor_axis = INT_MIN;
	AET->bres.minor_axis = ~INT_MAX;

	/* Initialize the Edge Table. */
	ET->scanlines.next = (ScanLineList *)NULL;
	//    ET->ymax = INT_MIN;
	ET->ymax = ~INT_MAX;
	ET->ymin = INT_MAX;
	pSLLBlock->next = (ScanLineListBlock *)NULL;

	PrevPt = &pts[count-1];

	/* For each vertex in the array of points.
	 * In this loop we are dealing with two vertices at
	 * a time -- these make up one edge of the polygon. */
	while (count--) {
		CurrPt = pts++;

		/* Find out which point is above and which is below. */
		if (PrevPt->y > CurrPt->y) {
			bottom = PrevPt, top = CurrPt;
			pETEs->ClockWise = 0;
		}
		else {
			bottom = CurrPt, top = PrevPt;
			pETEs->ClockWise = 1;
		}

		/* Don't add horizontal edges to the Edge table. */
		if (bottom->y != top->y) {
			pETEs->ymax = bottom->y-1;  /* -1 so we don't get last scanline */
			
			/* Initialize integer edge algorithm */
			dy = bottom->y - top->y;
			BRESINITPGONSTRUCT(dy, top->x, bottom->x, pETEs->bres);
			
			XMI_stubs_insertedgeinet(vis, ET, pETEs, top->y, &pSLLBlock, &iSLLBlock);
			ET->ymax = IMAX(ET->ymax, PrevPt->y);
			ET->ymin = IMIN(ET->ymin, PrevPt->y);
			pETEs++;
		}
		
		PrevPt = CurrPt;
	}
}

/*
 *     loadAET
 *
 *     This routine moves EdgeTableEntries from the
 *     EdgeTable into the Active Edge Table,
 *     leaving them sorted by smaller x coordinate.
 *
 */
void XMI_stubs_loadaet(ggi_visual_t vis, EdgeTableEntry *AET, EdgeTableEntry *ETEs)
{
	EdgeTableEntry *pPrevAET;
	EdgeTableEntry *tmp;

	XMIDPRINT_CORE("XMI_stubs_loadaet()\n");
	
	pPrevAET = AET;
	AET = AET->next;
	while (ETEs) {
		while (AET && (AET->bres.minor_axis < ETEs->bres.minor_axis)) {
			pPrevAET = AET;
			AET = AET->next;
		}
		tmp = ETEs->next;
		ETEs->next = AET;
		if (AET)
		  AET->back = ETEs;
		ETEs->back = pPrevAET;
		pPrevAET->next = ETEs;
		pPrevAET = ETEs;

		ETEs = tmp;
	}
}

/*
 *     computeWAET
 *
 *     This routine links the AET by the
 *     nextWETE (winding EdgeTableEntry) link for
 *     use by the winding number rule.  The final 
 *     Active Edge Table (AET) might look something
 *     like:
 *
 *     AET
 *     ----------  ---------   ---------
 *     |ymax    |  |ymax    |  |ymax    | 
 *     | ...    |  |...     |  |...     |
 *     |next    |->|next    |->|next    |->...
 *     |nextWETE|  |nextWETE|  |nextWETE|
 *     ---------   ---------   ^--------
 *         |                   |       |
 *         V------------------->       V---> ...
 *
 */
void XMI_stubs_computewaet(ggi_visual_t vis, EdgeTableEntry *AET)
{
	EdgeTableEntry *pWETE;
	int inside = 1;
	int isInside = 0;

	XMIDPRINT_CORE("XMI_stubs_computewaet()\n");
	
	AET->nextWETE = (EdgeTableEntry *)NULL;
	pWETE = AET;
	AET = AET->next;
	while (AET) {
		if (AET->ClockWise)
		  isInside++;
		else
		  isInside--;
		
		if ((!inside && !isInside) || (inside && isInside)) {
			pWETE->nextWETE = AET;
			pWETE = AET;
			inside = !inside;
		}
		AET = AET->next;
	}
	pWETE->nextWETE = (EdgeTableEntry *)NULL;
}

/*
 *     InsertionSort
 *
 *     Just a simple insertion sort using
 *     pointers and back pointers to sort the Active
 *     Edge Table.
 *
 */
int XMI_stubs_insertionsort(ggi_visual_t vis, EdgeTableEntry *AET)
{
	EdgeTableEntry *pETEchase;
	EdgeTableEntry *pETEinsert;
	EdgeTableEntry *pETEchaseBackTMP;
	int changed = 0;
	
	XMIDPRINT_CORE("XMI_stubs_insertionsort()\n");
	
	AET = AET->next;
	while (AET) {
		pETEinsert = AET;
		pETEchase = AET;
		while (pETEchase->back->bres.minor_axis > AET->bres.minor_axis)
		  pETEchase = pETEchase->back;

		AET = AET->next;
		if (pETEchase != pETEinsert) {
			pETEchaseBackTMP = pETEchase->back;
			pETEinsert->back->next = AET;
			if (AET)
			  AET->back = pETEinsert->back;
			pETEinsert->next = pETEchase;
			pETEchase->back->next = pETEinsert;
			pETEchase->back = pETEinsert;
			pETEinsert->back = pETEchaseBackTMP;
			changed = 1;
		}
	}
	return changed;
}

/* Clean up our act. */
void XMI_stubs_freestorage(ggi_visual_t vis, ScanLineListBlock *pSLLBlock)
{
	ScanLineListBlock *tmpSLLBlock;

	XMIDPRINT_CORE("XMI_stubs_freestorage()\n");
	
	while (pSLLBlock) {
		tmpSLLBlock = pSLLBlock->next;
		free(pSLLBlock);
		pSLLBlock = tmpSLLBlock;
	}
}
