/* This module provides several public functions: miNewPaintedSet(),
 * miAddSpansToPaintedSet(), miUniquifyPaintedSet(), miClearPaintedSet(),
 * miDeletePaintedSet().  They maintain a structure called a miPaintedSet,
 * which is essentially an array of SpanGroup structures, one per pixel
 * value.  A SpanGroup is essentially an unsorted list of Spans'.  A Spans
 * is a list of spans (i.e. horizontal ranges) of miPoints, sorted so that
 * the starting points have increasing y-values.  See mi_spans.h.
 * 
 * Internally, each libxmi drawing function paints to a miPaintedSet by
 * calling miAddSpansToPaintedSet() on one or more Spans's.  This function
 * adds a Spans to a miPaintedSet, being careful first to remove each from
 * the miPaintedSet each pixel in the Spans, if it has previously been
 * painted another color.  However, for efficiency it does not check
 * whether a pixel in the Spans has been previously painted the same color.
 * So while the drawing function is being called, the Spans in any one of
 * the PaintedSet's SpanGroups may overlap.  But different SpanGroups do
 * not overlap.  That is an invariant.
 * 
 * After all calls to miAddSpansToPaintedSet() are completed, duplicate
 * pixels are resolved by invoking miUniquifyPaintedSet().  That takes
 * place in the API wrappers in mi_api.c, just before the drawing function
 * returns.
 * 
 * The function miCopyPaintedSetToCanvas(), in mi_canvas.c, can copy the
 * contents of a miPaintedSet, i.e. its spans of painted miPoints, to a
 * miCanvas structure.  Sophisticated pixel merging is supported.  It would
 * be easy to write other functions that copy pixels out of a
 * miPaintedSet. */

/* Original version written by Joel McCormack, Summer 1989.  
   Hacked by Robert S. Maier, 1998-1999. */

#include <limits.h>
#include <string.h>
#include "stubsxmi.h"
#include "mi_api.h"
#include "xmi_debug.h"

/* spans in a Spans are sorted by y, so these give ymin, ymax for a nonempty Spans */
#define YMIN(spans) (spans->points[0].y)
#define YMAX(spans) (spans->points[spans->count-1].y)

/* internal functions */
static SpanGroup *XMI_stubs_newspangroup(ggi_visual_t vis);
static int XMI_stubs_uniquifyspansx(ggi_visual_t vis, const Spans *spans, miPoint *newPoints, unsigned int *newWidths);
static void XMI_stubs_addspanstospangroup(ggi_visual_t vis, const Spans *spans, SpanGroup *spanGroup);
static void XMI_stubs_deletespangroup(ggi_visual_t vis, SpanGroup *spanGroup);
static void XMI_stubs_quicksortspansx(ggi_visual_t vis, miPoint *points, unsigned int *widths, int numSpans);
static void XMI_stubs_subtractspans(ggi_visual_t vis, SpanGroup *spanGroup, const Spans *sub);
static void XMI_stubs_uniquifyspangroup(ggi_visual_t vis, SpanGroup *spanGroup);


/* The following functions are the public functions of this module. */

miPaintedSet *XMI_stubs_newpaintedset(ggi_visual_t vis) {
	miPaintedSet *paintedSet;
	
	XMIDPRINT_CORE("XMI_stubs_newpaintedset()\n");
	
	paintedSet = (miPaintedSet *)malloc(sizeof(miPaintedSet));
	paintedSet->size = 0;		/* slots allocated */
	paintedSet->ngroups = 0;	/* slots filled */
	paintedSet->groups = (SpanGroup **)NULL;
	return paintedSet;
}

/* Add a Spans to a miPaintedSet's SpanGroup for a specified pixel values,
 * and also subtract it from the SpanGroups for all other pixel values. */
void XMI_stubs_addspanstopaintedset(ggi_visual_t vis, Spans *spans, miPaintedSet *paintedSet, miPixel pixel) {
	int found = 0;
	int i;
	SpanGroup *spanGroup;

	XMIDPRINT_CORE("XMI_stubs_addspanstopaintedset()\n");

	if (spans->count == 0)
	  return;
	
	for (i = 0; i < paintedSet->ngroups; i++) {
		miPixel stored_pixel;
		
		stored_pixel = paintedSet->groups[i]->pixel;
		if (stored_pixel == pixel) {
			found = 1;		/* have a spanGroup for this pixel value */
			break;
		}
	}
	
	if (!found){
		if (paintedSet->ngroups == paintedSet->size) {
		  /* Expand array of SpanGroups */
			int old_size = paintedSet->size;
			int new_size = 2 * (old_size + 8);
			
			if (old_size == 0)
			  paintedSet->groups = (SpanGroup **)malloc(new_size * sizeof(SpanGroup *));
			else
			  paintedSet->groups = (SpanGroup **)realloc(paintedSet->groups, new_size * sizeof(SpanGroup *));
			paintedSet->size = new_size;
		}

		/* Create a SpanGroup for this pixel value */
		i = paintedSet->ngroups;
		paintedSet->groups[i] = XMI_stubs_newspangroup(vis);
		paintedSet->groups[i]->pixel = pixel;
		paintedSet->ngroups++;
	}
  
	spanGroup = paintedSet->groups[i];
	XMI_stubs_addspanstospangroup(vis, spans, spanGroup);
	
	/* Subtract Spans from all other SpanGroups */
	for (i = 0; i < paintedSet->ngroups; i++) {
		SpanGroup *otherGroup;
		
		otherGroup = paintedSet->groups[i];
		if (otherGroup == spanGroup)
		  continue;
		XMI_stubs_subtractspans(vis, otherGroup, spans);
	}
	
}

/* Deallocate all of a miPaintedSet's SpanGroups, including the points and
 * width arrays that are part of its component Spans's.  So it will
 * effectively become the empty set, as if it had been newly created. */
void XMI_stubs_clearpaintedset(ggi_visual_t vis, miPaintedSet *paintedSet)
{
	int i;

	XMIDPRINT_CORE("XMI_stubs_clearpaintedset()\n");

	if (paintedSet == (miPaintedSet *)NULL)
	  return;

	for (i = 0; i < paintedSet->ngroups; i++)
	  XMI_stubs_deletespangroup(vis, paintedSet->groups[i]);
	if (paintedSet->size > 0)
	  free(paintedSet->groups);
	paintedSet->size = 0;		/* Slots allocated */
	paintedSet->ngroups = 0;	/* Slots filled */
}

/* Deallocate a miPaintedSet, including the points and width arrays that
 * are part of its component Spans's. */
void XMI_stubs_deletepaintedset(ggi_visual_t vis, miPaintedSet *paintedSet)
{
	int i;

	XMIDPRINT_CORE("XMI_stubs_deletepaintedset()\n");

	if (paintedSet == (miPaintedSet *)NULL)
	  return;

	for (i = 0; i < paintedSet->ngroups; i++)
	  XMI_stubs_deletespangroup(vis, paintedSet->groups[i]);
	
	if (paintedSet->size > 0)
	  free(paintedSet->groups);
	free(paintedSet);
}

/* `Uniquify' a miPaintedSet, i.e. uniquify each of its SpanGroups (see below). */
void XMI_stubs_uniquifypaintedset(ggi_visual_t vis, miPaintedSet *paintedSet)
{
	int i;

	XMIDPRINT_CORE("XMI_stubs_uniqifypaintedset()\n");

	if (paintedSet == (miPaintedSet *)NULL)
	  return;

	for (i = 0; i < paintedSet->ngroups; i++)
	  if (paintedSet->groups[i]->count > 0)
	    XMI_stubs_uniquifyspangroup(vis, paintedSet->groups[i]);
}

/* Create and initialize a SpanGroup, i.e. an unsorted list of Spans's. */
static SpanGroup *XMI_stubs_newspangroup(ggi_visual_t vis)
{
	SpanGroup *spanGroup;

	XMIDPRINT_CORE("XMI_stubs_newspangroup()\n");

	spanGroup = (SpanGroup *)malloc (sizeof(SpanGroup));
	spanGroup->size = 0;			/* Slots allocated */
	spanGroup->count = 0;			/* Slots filled */
	spanGroup->group = (Spans *)NULL;	/* Slots for Spans's */
	spanGroup->ymin = INT_MAX;		/* Min over slots */
	spanGroup->ymax = INT_MIN;		/* Max over slots */

	return spanGroup;
}

/* Add a Spans to a SpanGroup, by tacking it on the end; update SpanGroup's ymin, ymax. */
static void XMI_stubs_addspanstospangroup(ggi_visual_t vis, const Spans *spans, SpanGroup *spanGroup)
{
	int ymin, ymax;

	XMIDPRINT_CORE("XMI_stubs_addspanstospangroup()\n");

	if (spans->count == 0)
	{
		return;
	}
	
	if (spanGroup->size == spanGroup->count) 
	  /* Expand SpanGroup */
	{
		spanGroup->size = (spanGroup->size + 8) * 2;
		spanGroup->group = (Spans *)realloc(spanGroup->group, sizeof(Spans) * spanGroup->size);
	}
	
  
	/* Tack Spans onto end of SpanGroup, update SpanGroup's ymin and ymax */
	spanGroup->group[spanGroup->count] = *spans;
	(spanGroup->count)++;
	ymin = YMIN(spans);
	if (ymin < spanGroup->ymin) 
	  spanGroup->ymin = ymin;
	ymax = YMAX(spans);
	if (ymax > spanGroup->ymax) 
	{
		spanGroup->ymax = ymax;
	}
}

/* Delete a SpanGroup, including the point and width arrays that are part of each Spans. */
static void XMI_stubs_deletespangroup(ggi_visual_t vis, SpanGroup *spanGroup) {
	int i;

	XMIDPRINT_CORE("XMI_stubs_deletespangroup()\n");

	if (spanGroup == (SpanGroup *)NULL)
	  return;

	for (i = 0; i < spanGroup->count; i++) {
		/*  free spanGroup->group[i], which is a Spans */
		free(spanGroup->group[i].points);
		free(spanGroup->group[i].widths);
	}
	
	if (spanGroup->group)
	  free(spanGroup->group);
	free(spanGroup);
}

/* Subtract a Spans from a SpanGroup, i.e. from each of its Spans's; update SpanGroup's ymin, ymax. */
static void XMI_stubs_subtractspans(ggi_visual_t vis, SpanGroup *spanGroup, const Spans *sub) {
	int		i, subCount, spansCount;
	int		ymin, ymax, xmin, xmax;
	Spans		*spans;
	miPoint	*subPt, *spansPt;
	unsigned int	*subWid, *spansWid;
	int		extra;
	int		gross_change = 0;
	
	XMIDPRINT_CORE("XMI_stubs_subtractspans()\n");
   
   if (sub->count == 0)		/* Nothing to do */
     return;

	/* y range of Spans to be subtracted */
	ymin = YMIN(sub);
	ymax = YMAX(sub);

	/* Loop through all Spans's in SpanGroup */
	spans = spanGroup->group;
	for (i = spanGroup->count; i > 0; i--, spans++) {
		/* Look only at Spans's with y ranges that overlap with `sub' */
		if (YMIN(spans) <= ymax && ymin <= YMAX(spans)) {
			/* Count, start points, and widths for `sub' */
			subCount = sub->count;
			subPt = sub->points;
			subWid = sub->widths;

			/* Count, start points, and widths for current Spans */
			spansCount = spans->count;
			spansPt = spans->points;
			spansWid = spans->widths;

			extra = 0;		/* Extra span slots available in Spans */
			for (;;) {
			  /* Look at pairs of spans, one from each Spans, that have the
			   * same value for y (break out when one or the other Spans is exhausted) */
				while (spansCount && spansPt->y < subPt->y) {
					spansPt++;  
					spansWid++; 
					spansCount--;
				}
				if (!spansCount)
				  break;
				while (subCount && subPt->y < spansPt->y) {
					subPt++;
					subWid++;
					subCount--;
				}
				if (!subCount)
				  break;
				
				if (subPt->y == spansPt->y) {
				  /* The two spans are at same y value, analyse in detail */
					xmin = subPt->x;
					xmax = xmin + (int)(*subWid); /* Just right of sub span */
					if (xmin >= spansPt->x + (int)(*spansWid) || spansPt->x >= xmax) {
					  /* Non-overlapping, do nothing */
						;
					}
					else if (xmin <= spansPt->x) {
					  /* Span to be subtracted begins at the same point, or to the left */
						if (xmax >= spansPt->x + (int)(*spansWid)) {
						  /* Span to be subtracted ends at the same point,
						   * or to the right; delete this span by downshifting */
							memmove(spansPt, spansPt + 1, sizeof(miPoint) * (spansCount - 1));
							memmove(spansWid, spansWid + 1, sizeof(unsigned int) * (spansCount - 1));
							spansPt--;
							spansWid--;
							spans->count--;
							extra++;
							gross_change = 1; /* Span vanished */
						}
						else {
						  /* Span to be subtracted ends to the left of this
						   * one's ending point; alter ending point and width */
							*spansWid = *spansWid - (unsigned int)(xmax - spansPt->x);
							spansPt->x = xmax;
						}
					}
					else {
					  /* Span to be subtracted overlaps with this one, and
					   * begins to the right of this one */
						if (xmax >= spansPt->x + (int)*spansWid)
						  /* Span to be subtracted ends at the same point, or
						   * to the right; just update width */
						  *spansWid = (unsigned int)(xmin - spansPt->x);
						else {
						  /* Hard case: must split the span */
#define EXTRA 8
							if (extra == 0) {
							  /* Reallocate; create EXTRA new span slots */
								miPoint *newPt;
								unsigned int *newwid;
								
								newPt = (miPoint *)realloc(spans->points, 
											   (spans->count + EXTRA) * 
											   sizeof(miPoint));
								spansPt = newPt + (spansPt - spans->points);
								spans->points = newPt;
								newwid = (unsigned int *)realloc(spans->widths, 
												 (spans->count + EXTRA) * 
												 sizeof(unsigned int));
								spansWid = newwid + (spansWid - spans->widths);
								spans->widths = newwid;
								extra = EXTRA;
							}

							/* Downshift; create two new spans as replacement */
							memmove(spansPt + 1, spansPt, sizeof(miPoint) * spansCount);
							memmove(spansWid + 1, spansWid, sizeof(unsigned int) * spansCount);
							spans->count++;
							extra--;
							/* First new span */
							*spansWid = (unsigned int)(xmin - spansPt->x);
							spansWid++;
							spansPt++;
							/* Second new span */
							*spansWid = *spansWid - (unsigned int)(xmax - spansPt->x);
							spansPt->x = xmax;
						}
					}
				} /* End of same-value-of-y computations */

				/* On to next span in the Spans */
				spansPt++;  
				spansWid++; 
				spansCount--;
			}
		}
	}

	if (gross_change) {
	  /* At least one span vanished; recompute SpanGroup's ymin, ymax */
		ymax = INT_MIN;
		ymin = INT_MAX;

		/* Loop through all Spans's in SpanGroup */
		spans = spanGroup->group;
		for (i = spanGroup->count; i > 0; i--, spans++) {
			int ymin_spans, ymax_spans;

			ymin_spans = YMIN(spans);
			ymax_spans = YMAX(spans);
			if (ymin_spans < ymin)
			  ymin = ymin_spans;
			if (ymax_spans > ymax)
			  ymax = ymax_spans;
		}
		
		spanGroup->ymin = ymin;
		spanGroup->ymax = ymax;      
	}
}

/* `Uniquify' a SpanGroup: merge all its Spans's into a single Spans, which
 *  will be sorted on x as well as on y. */
static void XMI_stubs_uniquifyspangroup(ggi_visual_t vis, SpanGroup *spanGroup) {
   int    i;
	Spans  *spans;
	Spans  *yspans;
	int    *ysizes;
	int    ymin, ylength;

	/* The new single Spans */
	miPoint *points;
	unsigned int *widths;
	int count;

	XMIDPRINT_CORE("XMI_stubs_uniquifyspangroup()\n");

	if (spanGroup->count == 0) 
	  return;

   /* Special case: ymin > ymax, so the Spans's in the SpanGroup, no matter
    * how numerous, must be empty (and cannot contain point or width arrays).
    */
   if (spanGroup->ymin > spanGroup->ymax)
     {
	spanGroup->count = 0;
	return;
     }
   
	/* Yuck.  Gross.  Radix sort into y buckets, then sort x and uniquify
	 * This seems to be the fastest thing to do.  I've tried sorting on
	 * both x and y at the same time rather than creating into all those
	 * y buckets, but it was somewhat slower. */
  
	ymin    = spanGroup->ymin;
	ylength = spanGroup->ymax - ymin + 1;
  
	/* Allocate Spans's for y buckets (one Spans for every scanline);
	 * ysizes[] is number of allocated Span slots in each bucket */
	yspans = (Spans *)malloc (ylength * sizeof(Spans));
	ysizes = (int *)malloc (ylength * sizeof(int));
	for (i = 0; i < ylength; i++) {
		ysizes[i]        = 0;
		yspans[i].count  = 0;
		yspans[i].points = (miPoint *)NULL;
		yspans[i].widths = (unsigned int *)NULL;
	}
  
	/* Go through every single span and put it into the correct y bucket */
	count = 0;
	for (i = 0, spans = spanGroup->group; i < spanGroup->count; i++, spans++) {
		int j, index;
		
		for (j = 0, points = spans->points, widths = spans->widths; j < spans->count; j++, points++, widths++) {
			index = points->y - ymin;
			if (index >= 0 && index < ylength) { /* paranoia */
				Spans *newspans = &(yspans[index]);
				
				if (newspans->count == ysizes[index]) {
				  /* expand bucket arrays by reallocating */
					ysizes[index] = (ysizes[index] + 8) * 2;
					newspans->points = (miPoint *)realloc(newspans->points, ysizes[index] * sizeof(miPoint));
					newspans->widths = (unsigned int *)realloc(newspans->widths, ysizes[index] * sizeof(unsigned int));
				}
				newspans->points[newspans->count] = *points;
				newspans->widths[newspans->count] = *widths;
				(newspans->count)++;
			} /* If y value of span in range */
		} /* For j through spans */
      
		count += spans->count;
	} /* For i through Spans */
	free(ysizes);
  
	/* Now sort each bucket by x and uniquify it into new Spans */
	points = (miPoint *)malloc(count * sizeof(miPoint));
	widths = (unsigned int *)malloc(count * sizeof(unsigned int));
	count = 0;
	for (i = 0; i < ylength; i++) {
		int ycount = yspans[i].count;
		
		if (ycount > 0) {
			if (ycount > 1) {
			  /* Sort the >1 spans at this value of y */
				XMI_stubs_quicksortspansx(vis, yspans[i].points, yspans[i].widths, ycount);
				count += XMI_stubs_uniquifyspansx (vis, &(yspans[i]), &(points[count]), &(widths[count]));
			} 
			else {
			  /* just a single span at this value of y */
				points[count] = yspans[i].points[0];
				widths[count] = yspans[i].widths[0];
				count++;
			}
			free(yspans[i].points);
			free(yspans[i].widths);
		}
	}
	free(yspans);
	
	/* free SpanGroup's original Spans's, including Span arrays */
	for (i = 0; i < spanGroup->count; i++) {
		free(spanGroup->group[i].points);
		free(spanGroup->group[i].widths);
	}

	/* SpanGroup now has only a single Spans */
	spanGroup->count = 1;
	spanGroup->group[0].points = points;
	spanGroup->group[0].widths = widths;
	spanGroup->group[0].count = count;
}


/* Sort each span in a Spans by x.  Called only if numSpans > 1. */
static void XMI_stubs_quicksortspansx(ggi_visual_t vis, miPoint *points, unsigned int *widths, int numSpans)
{
	int	 x;
	int	 i, j, m;
	miPoint *r;

	XMIDPRINT_CORE("XMI_stubs_quicksortspansx()\n");
	
#define ExchangeSpans(a, b)				    \
{							    \
miPoint tpt;	     					    \
unsigned int tw;					    \
\
tpt = points[a]; points[a] = points[b]; points[b] = tpt;    \
tw = widths[a]; widths[a] = widths[b]; widths[b] = tw;  \
}

	do {
		if (numSpans < 9) {
		  /* do insertion sort */
			int xprev;
			
			xprev = points[0].x;
			i = 1;
			do {			/* while i != numSpans */
				x = points[i].x;
				if (xprev > x) {
					/* points[i] is out of order.  Move into proper location. */
					miPoint tpt;
					unsigned int tw;
					int k;
					
					for (j = 0; x >= points[j].x; j++) { 
					}
					tpt = points[i];
					tw  = widths[i];
					for (k = i; k != j; k--) {
						points[k] = points[k-1];
						widths[k] = widths[k-1];
					}
					points[j] = tpt;
					widths[j] = tw;
					x = points[i].x;
				} /* if out of order */
				xprev = x;
				i++;
			} while (i != numSpans);
			
			/* end of insertion sort */
			return;
		}
		
		/* Choose partition element, stick in location 0 */
		m = numSpans / 2;
		if (points[m].x > points[0].x)
		  ExchangeSpans(m, 0);
		if (points[m].x > points[numSpans-1].x) 
		  ExchangeSpans(m, numSpans-1);
		if (points[m].x > points[0].x)
		  ExchangeSpans(m, 0);
		x = points[0].x;
		
		/* Partition array */
		i = 0;
		j = numSpans;
		do {
			r = &(points[i]);
			do {
				r++;
				i++;
			} 
			while (i != numSpans && r->x < x)
			  ;
			r = &(points[j]);
			do {
				r--;
				j--;
			} 
			while (x < r->x);
			if (i < j) ExchangeSpans(i, j);
		} 
		while (i < j);
		
		/* Move partition element back to middle */
		ExchangeSpans(0, j);
		
		/* Recurse */
		if (numSpans-j-1 > 1)
		  XMI_stubs_quicksortspansx(vis, &points[j+1], &widths[j+1], numSpans-j-1);
		numSpans = j;
	} while (numSpans > 1);
}

/* Sort an unordered list of spans by y, so that it becomes a Spans. */
void XMI_stubs_quicksortspansy(ggi_visual_t vis, miPoint *points, unsigned int *widths, int numSpans)
{
	int	 y;
	int	 i, j, m;
	miPoint *r;

	XMIDPRINT_CORE("XMI_stubs_quicksortspansy()\n");

	if (numSpans <= 1)		/* nothing to do */
	  return;

#define ExchangeSpans(a, b)				    \
  {							    \
    miPoint tpt;	     					    \
    unsigned int tw;					    \
								\
    tpt = points[a]; points[a] = points[b]; points[b] = tpt;    \
    tw = widths[a]; widths[a] = widths[b]; widths[b] = tw;  \
  }

	do {
		if (numSpans < 9) {
		  /* do insertion sort */
			int yprev;
			
			yprev = points[0].y;
			i = 1;
			do {			/* while i != numSpans */
				y = points[i].y;
				if (yprev > y) {
					/* points[i] is out of order.  Move into proper location. */
					miPoint tpt;
					unsigned int tw;
					int k;
					
					for (j = 0; y >= points[j].y; j++) { 
					}
					tpt = points[i];
					tw  = widths[i];
					for (k = i; k != j; k--) {
						points[k] = points[k-1];
						widths[k] = widths[k-1];
					}
					points[j] = tpt;
					widths[j] = tw;
					y = points[i].y;
				} /* if out of order */
				yprev = y;
				i++;
			} while (i != numSpans);
			
			/* end of insertion sort */
			return;
		}
		
		/* Choose partition element, stick in location 0 */
		m = numSpans / 2;
		if (points[m].y > points[0].y)
		  ExchangeSpans(m, 0);
		if (points[m].y > points[numSpans-1].y) 
		  ExchangeSpans(m, numSpans-1);
		if (points[m].y > points[0].y)
		  ExchangeSpans(m, 0);
		y = points[0].y;
		
		/* Partition array */
		i = 0;
		j = numSpans;
		do {
			r = &(points[i]);
			do {
				r++;
				i++;
			} 
			while (i != numSpans && r->y < y)
			  ;
			r = &(points[j]);
			do {
				r--;
				j--;
			} 
			while (y < r->y);
			if (i < j) ExchangeSpans(i, j);
		} 
		while (i < j);
		
		/* Move partition element back to middle */
		ExchangeSpans(0, j);
		
		/* Recurse */
		if (numSpans-j-1 > 1)
		  XMI_stubs_quicksortspansy(vis, &points[j+1], &widths[j+1], numSpans-j-1);
		numSpans = j;
	} while (numSpans > 1);
}

/* Uniquify the spans in a Spans.  (Spans at each y value are assumed to
 * have been sorted on x, perhaps by calling miQuickSortSpansX() above.)
 * Also, create a new Spans: stash the uniquified spans into the previously
 * allocated arrays newPoints and newWidths.  Returns the number of unique
 * spans.  Called only if numSpans > 1. */
static int XMI_stubs_uniquifyspansx(ggi_visual_t vis, const Spans *spans, miPoint *newPoints, unsigned int *newWidths)
{
	int		newx1, newx2, oldpt, i, y;
	miPoint	*oldPoints;
	unsigned int	*oldWidths, *startNewWidths;

	XMIDPRINT_CORE("XMI_stubs_uniquifyspansx()\n");

	startNewWidths = newWidths;
	oldPoints = spans->points;
	oldWidths = spans->widths;
	y = oldPoints->y;
	newx1 = oldPoints->x;
	newx2 = newx1 + (int)(*oldWidths);

	for (i = spans->count - 1; i > 0; i--) {
		oldPoints++;
		oldWidths++;
		oldpt = oldPoints->x;
		if (oldpt > newx2) {
			/* write current span, start a new one */
			newPoints->x = newx1;
			newPoints->y = y;
			*newWidths = (unsigned int)(newx2 - newx1);
			newPoints++;
			newWidths++;
			newx1 = oldpt;
			newx2 = oldpt + (int)(*oldWidths);
		} 
		else {
			/* extend current span, if old extends beyond new */
			oldpt = oldpt + (int)(*oldWidths);
			if (oldpt > newx2)
			  newx2 = oldpt;
		}
	}
	
	/* write final span */
	newPoints->x = newx1;
	*newWidths = (unsigned int)(newx2 - newx1);
	newPoints->y = y;
	
	return (int)((newWidths - startNewWidths) + 1);
}
