/* This is the public include file for the miPaintedSet module, contained
 * in mi_spans.c. */

/* A Spans structure is a sorted list of spans, i.e. a list of point ranges
 * [xmin,xmax], sorted in increasing y order.  There may be more than one
 * span at a given y. */

typedef struct 
{
	unsigned int count;    	/* Number of spans		    */
	miPoint *points;	/* Pointer to list of start points  */
	unsigned int *widths;	/* Pointer to list of widths	    */
} Spans;

/* A SpanGroup is an unsorted list of Spans's, associated with a particular
 * pixel value.  
 *
 * A SpanGroup is allowed to include more than a single Spans because most
 * libxmi drawing functions write out multiple Spans's.
 */
typedef struct 
{
	miPixel	pixel;		/* pixel value				*/
	Spans   *group;		/* Spans slots				*/
	unsigned int size;     	/* Number of Spans slots allocated	*/
	unsigned int count;    	/* Number of Spans slots filled		*/
	int ymin, ymax;		/* min, max y values over all Spans's	*/
} SpanGroup;

/* A miPaintedSet structure is an array of SpanGroups, specifying the
 * partition into differently painted subsets.  There is at most one
 * SpanGroup for any pixel. */
typedef struct lib_miPaintedSet
{
	SpanGroup **groups;	/* SpanGroup slots			*/
	unsigned int size;     	/* Number of SpanGroup slots allocated	*/
	unsigned int ngroups;	/* Number of SpanGroup slots filled	*/
} miPaintedSet;

/* libxmi's low-level painting macro.  It `paints' a Spans, i.e. a list of
 * spans assumed to be in y-increasing order, to a miPaintedSet with a
 * specified pixel value.  To do this, it invokes the lower-level function
 * miAddSpansToPaintedSet() in mi_spans.c.
 * 
 * FIXME: We really need to ggiDrawHLine() here.
 * 
 * The passed point and width arrays should have been allocated on the
 * heap, since they will be eventually freed; e.g., when the miPaintedSet
 * is cleared or deallocated. */

#define MI_PAINT_SPANS(paintedSet, pixel, numSpans, ppts, pwidths) \
  {\
    Spans 	spanRec;\
    if (numSpans > 0) \
      { \
        spanRec.points = (ppts);\
        spanRec.widths = (pwidths);\
        spanRec.count = (numSpans);\
        XMI_stubs_addspanstopaintedset(vis, &spanRec, (paintedSet), (pixel));\
      } \
    else \
      { \
        free (ppts); \
        free (pwidths); \
      } \
    }

/* A wrapper for MI_PAINT_SPANS() that can be applied to a span array
 * (i.e. to a point and width array) that can't be freed, so must be
 * copied.  We try not to use this.
 */
#define MI_COPY_AND_PAINT_SPANS(paintedSet, pixel, nPts, FirstPoint, FirstWidth) \
  {\
    if ((nPts) > 0) \
      { \
        miPoint		*ppt, *pptInit, *oldppt; \
        unsigned int	*pwidth, *pwidthInit, *oldpwidth; \
        int		ptsCounter; \
        ppt = pptInit = (miPoint *) malloc ((nPts) * sizeof (miPoint));\
        pwidth = pwidthInit = (unsigned int *) malloc ((nPts) * sizeof (unsigned int));\
        oldppt = FirstPoint;\
        oldpwidth = FirstWidth;\
        for (ptsCounter = (nPts); --ptsCounter >= 0; )\
          {\
            *ppt++ = *oldppt++;\
            *pwidth++ = *oldpwidth++;\
          }\
        MI_PAINT_SPANS(paintedSet, pixel, (nPts), pptInit, pwidthInit)\
      } \
  }

/* miPaintedSet manipulation routines (other than public) */
extern void XMI_stubs_addspanstopaintedset(ggi_visual_t vis, Spans *spans, miPaintedSet *paintedSet, miPixel pixel);
extern void XMI_stubs_quicksortspansy(ggi_visual_t vis, miPoint *points, unsigned int *widths, int numSpans);
extern void XMI_stubs_uniquifypaintedset(ggi_visual_t vis, miPaintedSet *paintedSet);
