/*
******************************************************************************

   X-Target library for libXMI.

   Copyright (C) 2000      Jon Taylor	[taylorj@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/


#include "stubsxmi.h"
#include "mi_api.h"
#include "xmi_debug.h"

void XMI_stubs_paintvisual(ggi_visual_t vis, const miGC *pGC, miPixel pixel, int n, const miPoint *ppt, const unsigned int *pwidth, miPoint offset);

/* Paint a list of spans, in a specified miPixel color, to 
 * a ggi_visual.  The spans must be in y-increasing order. */
void XMI_stubs_paintvisual(ggi_visual_t vis, const miGC *pGC, miPixel pixel, int n, const miPoint *ppt, const unsigned int *pwidth, miPoint offset)
{
	int i, ii;
	int xleft, xright, ybottom, ytop;
	int xstart, xend, xstart_clip, xend_clip, xoffset, yoffset, x, y;
	unsigned int width;
	ggi_mode mode;
	
	XMIDPRINT_CORE("XMI_stubs_paintvisual()\n");
	
	xoffset = offset.x;
	yoffset = offset.y;
	
	/* Compute bounds of destination drawable */
	ggiGetMode(vis, &mode);
	xleft = 0;
	ytop = 0;
	xright = mode.visible.x;
	ybottom = mode.visible.y;

	/* If source doesn't overlap with destination drawable, do nothing */
	if (ppt[0].y + yoffset > ybottom || ppt[n-1].y + yoffset < ytop)
	  return;
	
	for (i = 0; i < n; i++) {
		y = ppt[i].y + yoffset;
		if (y > ybottom)
		  return;			/* No more spans will be painted */
		
		if (y >= ytop) {
			
			ggi_pixel workingpixel, destpixel;
			ggi_color workingcolor, destcolor;
			void *op1, *op2;
			uint type;
			int mask = 0;
			
			width = pwidth[i];
			xstart = ppt[i].x + xoffset;
			xend = xstart + (int)width - 1;
			
			xstart_clip = IMAX(xstart,xleft);
			xend_clip = IMIN(xend,xright);

			/* Use ggiDrawHLine() for possible acceleration */
			/* if no blend stages are active (solid-fill). */
			if (pGC->numInBlendStageList == 0) {
				XMIDPRINT_CORE("No blend stages, doing solid hline\n");
				ggiSetGCForeground(vis, pixel);
				ggiDrawHLine(vis, xstart_clip, y, xend_clip - xstart_clip + 1);
			}
			else 
			{
				for (x = xstart_clip; x <= xend_clip; x++) { /* may be empty */
					
					workingpixel = pixel;					
					
					ggiUnmapPixel(vis, workingpixel, &workingcolor);
					ggiGetPixel(vis, x, y, &destpixel);
					ggiUnmapPixel(vis, destpixel, &destcolor);
					
					mask = 0;
					
					/* Loop through all blend stages, evaluating each */
					XMIDPRINT_CORE("Evaluating %d blend stages...\n", pGC->numInBlendStageList);
					for (ii = 0; ii < pGC->numInBlendStageList; ii++) {
						XMIDPRINT_CORE("Stage %d\n", ii);
						
						op1 = pGC->blendstages[ii].op1;
						op2 = pGC->blendstages[ii].op2;
						type = pGC->blendstages[ii].type;
						
						ggiUnmapPixel(vis, workingpixel, &workingcolor);
						
						switch (type) {
							
						    case MI_BLEND_COLORKEY: 
							XMIDPRINT_CORE("MI_BLEND_COLORKEY:\n");
							{
								/* Pixels of type OP1 (ggi_color) are not drawn */
								ggi_pixel *keypixel = (ggi_pixel *)op1;
								
								mask = (*keypixel == destpixel);
							}
							break;
							
						    case MI_BLEND_TEXTURE_ROP:
							XMIDPRINT_CORE("MI_BLEND_TEXTURE_ROP:\n");
							{
								/* Pixels in the destination visual are replaced by */
								/* pixels in the source texture map (OP2) according */
								/* to the ruleset defined by the rasterop (OP1). */
								miRopType *roptype = (miRopType *)op1;
								miPixmap *texturemap = (miPixmap *)op2;
								
								switch (*roptype)
								{
								    case ROP_PAINT:
									XMIDPRINT_CORE("ROP_PAINT:\n");
									ggiGetPixel(texturemap->vis, 
										    x % texturemap->rect.width, 
										    y % texturemap->rect.height, 
										    &workingpixel);
									break;
								    default:
									XMIDPRINT_CORE("Unknown raster operation (%d)!\n", roptype);
									break;
								}
							}
							break;
							
						    case MI_BLEND_ALPHA_BUFFER:
							XMIDPRINT_CORE("MI_BLEND_ALPHA_BLEND:\n");
							{
								/* Alpha blend - write from the source to the destination, */
								/* using a blend operation (OP1) and optionally through an */
								/* 8-bit alpha buffer pixmap (OP2) */
								uint *blendop = (uint *)op1;
								
								switch (*blendop) {
								    case BLEND_SC_OR_DC:	/* Blend source and destination */
									XMIDPRINT_CORE("BLEND_SC_OR_D:\n");
									workingpixel |= destpixel;
									break;
									
								    case BLEND_SC_AND_DC:	/* Blend source and destination */
									XMIDPRINT_CORE("BLEND_SC_AND_D:\n");
									workingpixel |= destpixel;
									break;
									
								    case BLEND_SC_AND_AB:	/* Blend source & alpha buffer values */
									XMIDPRINT_CORE("BLEND_SC_AND_AB:\n");
									{
										miPixmap *abuffer = (miPixmap *)op2;
										ggi_color col = workingcolor;
										ggi_pixel alphaval;
										
										XMIDPRINT_CORE("workingpixel(before) = %d\n",
											       workingpixel);
										
										ggiGetPixel(abuffer->vis, 
											    x % abuffer->rect.width, 
											    y % abuffer->rect.height, 
											    &alphaval);
										
										XMIDPRINT_CORE("alphaval = %d\n",
											       alphaval);
										
										col.r &= (alphaval << 8);
										col.g &= (alphaval << 8);
										col.b &= (alphaval << 8);
										col.a = (alphaval << 8);
										
										XMIDPRINT_CORE("workingpixel(after) = %d\n",
											       workingpixel);
										
										workingpixel = ggiMapColor(vis, &col);
									}
									break;

								    case BLEND_SC_AND_AC:	/* Blend source & alpha constant values */
									XMIDPRINT_CORE("BLEND_SC_AND_AC:\n");
									{
										uint16 alphaval = *((uint16 *)op2);
										ggi_color col = workingcolor;
										
										col.r &= alphaval;
										col.g &= alphaval;
										col.b &= alphaval;
										col.a = alphaval;
										
										workingpixel = ggiMapColor(vis, &col);
									}
									break;
									
								    case BLEND_SC_AND_STB:	/* Blend source & stencil buffer */
									XMIDPRINT_CORE("BLEND_SC_AND_STB:\n");
									{
										miPixmap *sbuffer = (miPixmap *)op2;
										ggi_pixel stencilVal;
										ggi_color col = workingcolor;

										ggiGetPixel(sbuffer->vis, x, y, &stencilVal);
										
										mask = (stencilVal == 0);
									}
									break;

								    default:
									XMIDPRINT_CORE("Unknown blend operation (%d)!\n", *blendop);
									break;
								}
							}
							break;
							
						    case MI_BLEND_NULL:
							XMIDPRINT_CORE("XMI_BLEND_NULL:\n");
							break;
							
						    default: 
							XMIDPRINT_CORE("Unknown blend stage type (%d)!\n", 
								       pGC->blendstages[ii].type);
							break;
						}
					}

					if (!mask)
					  ggiPutPixel(vis, x, y, workingpixel);
				}
			}
			
		} /* end if */
	} /* end for y in ... */	
}

/* Copy a miPaintedSet to an ggi_visual.  The miPaintedSet is assumed to have
 * been uniquified (see mi_spans.c), which is the case after any of the
 * eight core drawing functions in the libxmi API has been invoked.  So
 * there is at most one Spans per SpanGroup (i.e., Spans #0), and copying
 * pixels out of it is trivial. */
void XMI_stubs_copypaintedsettovisual(ggi_visual_t vis, const miGC *pGC, const miPaintedSet *paintedSet, miPoint offset)
{
	int i;
	
	XMIDPRINT_CORE("XMI_stubs_copypaintedsettovisual()\n");
	
	/* For each pixel color, the initial Spans in the corresponding SpanGroup
	 * is effectively a list of spans, in y-increasing order.  That is, it's
	 * a list of starting points and a corresponding list of widths; both are
	 * of length `count'. */

	for (i = 0; i < paintedSet->ngroups; i++)
	  if (paintedSet->groups[i]->group[0].count > 0)
	    XMI_stubs_paintvisual(vis, pGC, paintedSet->groups[i]->pixel,
				  paintedSet->groups[i]->group[0].count, 
				  paintedSet->groups[i]->group[0].points, 
				  paintedSet->groups[i]->group[0].widths, offset);
	
}
