/* $Id: speedometer.c,v 1.3 2001/08/17 17:22:10 cegger Exp $
******************************************************************************

   Speedometer for any target

   Author: 2000 Christoph Egger [Christoph_Egger@t-online.de]

   This software is placed in the public domain and can be used freely
   for any purpose. It comes without any kind of warranty, either
   expressed or implied, including, but not limited to the implied
   warranties of merchantability or fitness for a particular purpose.
   Use it at your own risk. the author is not responsible for any damage
   or consequences raised by use or inability to use this program.

******************************************************************************

   The base code comes from the demo.c written by Jon Taylor [taylorj@ggi-project.org]

   This is a demonstration of LibXMI's functions and can be used as a 
   reference programming example. Thus it is heavily commented to explain
   you every single step to make your own applications.

******************************************************************************
*/

/* This is needed for the HAVE_* macros */
#include "config.h"

#if 0
#ifndef HAVE_RANDOM
# define random		rand
# define srandom	srand
#endif
#endif

/* Include the necessary headers used for e.g. error-reporting */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

/* Include the LibGGI declarations */
#include <ggi/ggi.h>
#include "xmi.h"

/* We are running on a single primary visual which is made accessible
 * globally so subroutines like "waitabit" can access it.
 * See its allocation in main() for details. */
ggi_visual_t vis;


/* In case we were called with wrong parameters, give an explanation.
 */
void usage(const char *prog)
{
	fprintf(stderr, "Usage:\n\n"
		        "%s [--clip] [--target <target>] "
			"<xsize>x<ysize>[#<virtx>x<virty>]"
			"['['<bpp>']']\n\n"
		       "Example: %s 320x200[8]\n"
		"For textmode, try %s 80x25T\n",prog,prog,prog);
	exit(1);
}


/* Wait for a keypress. Shut down everything, if "q" is pressed.
 */
void waitabit(void)
{
	int key;

	/* Make sure the display is in sync with the drawing command queue. 
	 */
	ggiFlush(vis);

#if 0	/* Activate this to test waiting for max times. 
	 * It will cause the wait to not block indefinitely but only for
	 * 1 seconds.
         */
	{
		struct timeval tv={1,0};
		key=ggiEventPoll(vis,emKeyPress,&tv);
		if (! (key & emKeyPress)) return;
		key = ggiGetc(vis);
	}

#else
	/* ggiGetc will blocking-wait for a keypress. 
	 * we ignore modifier keys.
	 */
	do {
		key = ggiGetc(vis);

	} while ((key == GIIK_VOID) || (GII_KTYP(key) == GII_KT_MOD));
#endif

	if ((key == 'q') || (key == 'Q'))	/* Q pressed */
	{	
		ggiClose(vis);
		ggiExit();
		exit(0);
	}
}

/* Pixel value for white and black. See main() on how to get at it.
 */
ggi_pixel white;
ggi_pixel black;

/* Print the name of the current test in the top left corner.
 */
void TestName(const char *name)
{
	int left, right, top, bottom;
	int ch_width=8, ch_height=8;
	ggi_mode mode;

	/* Get the current mode. This is needed to determine the size of the
	 * visible area.
	 */
	ggiGetMode(vis, &mode);

	/* Get the font dimensions for ggiPuts.
	 */
	ggiGetCharSize(vis, &ch_width, &ch_height);
	
	/* Save away the old clipping area, and set one that allows to
	 * print the test name in the top left corner.
	 */
	ggiGetGCClipping(vis, &left, &top, &right, &bottom);
	ggiSetGCClipping(vis, 0, 0, mode.virt.x, ch_height);

	/* Now draw a nice black background. Note, that the rendered text
	 * already has a background color. But it is nicer, if the whole 
	 * line is black ...
	 */
	ggiSetGCForeground(vis, black);
	ggiDrawBox(vis, 0, 0, mode.virt.x, ch_height);

	/* Now set the colors of the text and render it.
	 */
	ggiSetGCBackground(vis, black);
	ggiSetGCForeground(vis, white);
	ggiPuts(vis, 0, 0, name);

	/* Flush commands.
	 */
	ggiFlush(vis);

	/* Restore the old clipping rectangle.
	 */
	ggiSetGCClipping(vis, left, top, right, bottom);
}

/* Timing helper routines 
*/
static struct timeval test_start_time;

void TestStart(void)
{
	ggCurTime(&test_start_time);
}

int TestTime(void)	/* result is in seconds */
{
	int seconds;
	int micros;

	struct timeval cur_time;

	ggCurTime(&cur_time);

	seconds = cur_time.tv_sec  - test_start_time.tv_sec;
	micros  = cur_time.tv_usec - test_start_time.tv_usec;

	if (micros < 0) {
		micros += 1000000;
		seconds--;
	}

	return seconds;
}


/* The main routine.
 * It will set up a graphics mode as requested on the commandline and 
 * do an extensive test using about all graphics primitives LibXMI
 * knows.
 */
int main(int argc, char **argv)
{
	/* First we define a bunch of variables we will access throughout the
	 * main() function. Most of them are pretty meaningless loop-counters
	 * and helper-variables.
	 */

	const char *prog;	/* Make an alias for the program name */
	
	/* This is an enum holding the requested type of graphics mode.
	 * See the mode setting code below for details.
	 */
	 
	ggi_graphtype type;

	/* This holds the palette we will soon set for GT_8BIT type
	 * graphics modes
	 */
	 
	ggi_color pal[256];

	/* The depth is used for chosing a graphics type.  it is mapped
	 * to the right GT_* by a switch-statement.
	 */
	 
	int depth=0;

	/* Demonstrate clipping ? */

	int doclip=0;

	/* Use a different target than the default ? */

	char *target_name=NULL;

	/* These hold the visible screen size (sx,sy) and
	 * the virtual size (vx,vy). On most targets you can 
	 * request a certain larger area on which the visible
	 * area can be placed.
	 */

	int sx,sy,vx,vy;
	
	/* Pretty meaningless helpers used everywhere. */

	int x,y,w,h,i,err;

	int sx_5, sx_12;   /* 1/5 and 1/12 respectively */
	int sy_5, sy_12;

	/* Yet another array of colors used for the color-
	 * mapping test
	 */

	ggi_color map[256];
		
	/* This is a struct containing visual and virtual screen size as
	 * well as the bpp/textmode information 
	 */

	ggi_mode mode = { /* This will cause the default mode to be set */
		1,                      /* 1 frame [???] */
		{GGI_AUTO,GGI_AUTO},    /* Default size */
		{GGI_AUTO,GGI_AUTO},    /* Virtual */
		{0,0},                  /* size in mm don't care */
		GT_AUTO,               /* Mode */
		{GGI_AUTO,GGI_AUTO}     /* Font size */
	};

	miGC *pGC;                /* graphics context */
	miBlendStage blendstages[4];
	miPaintedSet *paintedSet; /* opaque object to be painted */
	miPixel pixels[100];        /* pixel values for drawing and dashing */
	miPoint points[100];        /* 3 line segments in the polyline */
	unsigned int dashes[100];   /* length of `on' and `off' dashes */
	miPoint offset;           /* for miPaintedSet -> miCanvas transfer */
	miRectangle rect;                /* 1 arc to be drawn */
	miArc arc;

	/* Get the arguments from the command line. 
	 * Set defaults for optional arguments. */

	prog=*argv; argv++; argc--;
	if (strrchr(prog, '/') != NULL) prog=strrchr(prog, '/')+1;

	if ((argc > 0) &&
	    ((strcmp(*argv, "-h")==0) || 
	     (strcmp(*argv, "--help")==0))) {

		usage(prog);	/* This is not an allowed call format. */
		return 1;	/* Tell the user how to call and fail. */
	}

	if ((argc > 0) && 
	    ((strcmp(*argv, "-c") == 0) ||
	     (strcmp(*argv, "--clip") == 0))) {

		/* Enable clipping */
		doclip=1; argv++; argc--;
	}
	
	if ((argc > 1) && 
	    ((strcmp(*argv, "-t") == 0) ||
	     (strcmp(*argv, "--target") == 0))) {

		target_name=argv[1]; argv+=2; argc-=2;
	}
	
	if (argc > 1) {
		usage(prog);	/* This is not an allowed call format. */
		return 1;	/* Tell the user how to call and fail. */
	}
	
	/* Set up the random number generator. */
	srandom(time(NULL));

	/* Initialize the GGI library. This must be called before any other 
	 * GGI function. Otherwise behaviour is undefined.
	 */
	if (ggiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize LibGGI, exiting.\n",
			prog);
		exit(1);
	}
	
	if (xmiInit() != 0) {
		fprintf(stderr, "%s: unable to initialize LibXMI, exiting.\n",
			prog);
		exit(1);
	}

	if (target_name == NULL) {

		/* Open the default visual. This is automatically
		 * selected depending on the calling environment (Linux
		 * console, X, ...). You can explicitly set this in the
		 * GGI_DISPLAY environment variable.
		 */

		vis=ggiOpen(NULL);
	} else {

		/* Open a specific visual.  Note the NULL, which is used
		 * to mark the end of a *list* of LibGGI sub-library names
		 * to be loaded (although at present, it only makes sense 
		 * having either 0 or 1 names in the list).
		 */
		
		vis=ggiOpen(target_name, NULL);
	}

	if (vis == NULL) {
		fprintf(stderr,
			"%s: unable to open default visual, exiting.\n",
			prog);
		ggiExit();
		exit(1);
	}
	/* Using ASYNC mode can give great performance improvements on
	   some targets so it should be used as much as possible.
	   Turning ASYNC mode on right after ggiOpen() ensures best operation
	*/
	ggiSetFlags(vis, GGIFLAG_ASYNC);

	if (argc == 1) { /* A mode-string was given - so we parse it */
		if (ggiParseMode(*argv, &mode) != 0) {
			/* Error parsing mode */
			fprintf(stderr,
				"Error parsing mode: %s\n"
				"Valid modes are:\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames (T) [bpp] + XOFF + YOFF\n"
"  X x Y # XVIRT x YVIRT D dpp.x x dpp.y F frames [graphtype] + XOFF + YOFF\n\n"
"  Everything (!) can be omitted, all ommitted values default to GGI_AUTO,\n"
"  with the exception of a textmode with unspecified size which defaults\n"
"  to TEXT16.\n  Whitespace is ignored.\n"
				, *argv);
			exit(1);
		}
		/* mode now holds the requested visible 
		 * and virtual size of the screen. */
	}

	/* that's what we try. See what we get ... */
	fprintf(stderr, "Trying mode ");
	ggiFPrintMode(stderr, &mode);
	fprintf(stderr, "\n");

	/* Is the mode possible ? If not, a better one will be
	 * suggested. */

	ggiCheckMode(vis,&mode);

	fprintf(stderr, "Suggested mode ");
	ggiFPrintMode(stderr, &mode);
	fprintf(stderr, "\n");

	err = ggiSetMode(vis,&mode);   /* now try it. it *should* work! */
	
	/* Check if there were errors. Almost all LibGGI functions
	 * return 0 on success and an error code otherwise. No messing
	 * with errno, as this is not thread- friendly. The functions
	 * that are supposed to return a pointer normally return NULL on
	 * error. */

	if (err) {
		fprintf(stderr,"Can't set mode\n");
		ggiClose(vis);
		ggiExit();
		return 2;
	}


	/* Now we read back the set mode, as it might have been
	 * autoselected or changed. */

	type=mode.graphtype;
	vx=mode.virt.x;    vy=mode.virt.y;
	sx=mode.visible.x; sy=mode.visible.y;
	depth=GT_DEPTH(mode.graphtype);

	sx_5 = sx / 5; sx_12 = sx / 12;
	sy_5 = sy / 5; sy_12 = sy / 12;
	
	/* Set a colorful palette for the tests.
	   Please note that GGI always uses 16 bit color components,
	   so stretch the values accordingly when porting from DOS 
	   or other libs that make assumptions about palette size.

	   On some fixed-palette modes the ggiSetColorfulPalette()
	   call will fail.  We silently ignore that.
	 */
	 
	if (GT_SCHEME(mode.graphtype) == GT_PALETTE) {
		ggiSetColorfulPalette(vis);
		ggiGetPalette(vis, 0, 1<<depth, pal);
	}
	
	/* Attach the XMI extension to the visual */
	if (xmiAttach(vis) < 0) {
		ggiPanic("Unable to attach XMI extension to visual\n");
	}

	/* Find the colors "white" and "black". */
	map[0].r=0xFFFF; /* this is the highest intensity value for the red part. */
	map[0].g=0xFFFF;
	map[0].b=0xFFFF;
	
	white=ggiMapColor(vis, &map[0]);
	fprintf(stderr, "white=%d\n", white);

	map[0].r= map[0].g= map[0].b= 0x0;
	black=ggiMapColor(vis, &map[0]);
	fprintf(stderr, "black=%d\n", black);

	/* Set the drawing color to black and clear the screen.
	 * (The screen is cleared to black at setmode time, but the
	 * palette change above might have ruined that) */
	ggiSetGCForeground(vis, black);
	ggiFillscreen(vis);
	
	/* Set the drawing color to white and background color to black.
	 * Then display some text at 0/0.*/
	ggiSetGCForeground(vis, white);
	ggiSetGCBackground(vis, black);
	ggiPuts(vis,0,0,"Press any key to begin tests...");
	ggiFlush(vis);
	
	/* Wait for any keypress. 
	 * This is a blocking call which returns a 16 bit wide unicode
	 * character */
	ggiGetc(vis);

	if (doclip) {
		ggiDrawHLine(vis, 0, sy_5-1,  sx);
		ggiDrawHLine(vis, 0, sy-sy_5, sx);
		ggiDrawVLine(vis, sx_5-1,  0, sy);
		ggiDrawVLine(vis, sx-sx_5, 0, sy);
		ggiSetGCClipping(vis, sx_5, sy_5, sx-sx_5, sy-sy_5);
	}


	/* Here we turn on SYNC mode again beacuse we want to make sure
	 * the user sees the drawing as it happens.
	 * Note that in SYNC mode no ggiFlush() calls are necessary. */
	ggiRemoveFlags(vis, GGIFLAG_ASYNC);
	
	/* Create empty painted set */
	paintedSet = miNewPaintedSet(vis);
	
	offset.x = 0; offset.y = 0;

#define TEST_SETUP(name) \
	TestName(name); \
	waitabit(); \
	ggiSetGCForeground(vis,0); \
	ggiFillscreen(vis); \
	TestStart(); \
	printf("%s: ", name);	\
	for (i=0; TestTime() < 1; i++) { \
		int j, numpoints, numpixels, numblendstages; \
		ggi_color col; \
		numpixels = random() % 20 + 2; \
		numpoints = random() % 10 + 4; \
		numblendstages = 0; \
		pGC = miNewGC(vis, numpixels, pixels, numblendstages, blendstages); \
		for (j = 0; j < numpixels; j++) { \
			col.r = random() % 0xffff; \
			col.g = random() % 0xffff; \
			col.b = random() % 0xffff; \
			pGC->pixels[j] = ggiMapColor(vis, &col); \
		} \
		for (j = 0; j < numpoints; j++) { \
			points[j].x = random() % vx; \
			points[j].y = random() % vy; \
		} \
		dashes[0] = random() % 20; \
		dashes[1] = random() % 20; \
		x = random() % vx; \
		y = random() % vy; \
		w = random() % (vx-x) + 1; \
		h = random() % (vy-y) + 1; \
		miSetGCAttrib(vis, pGC, MI_GC_LINE_STYLE, MI_LINE_ON_OFF_DASH); \
		miSetGCDashes(vis, pGC, 2, dashes, 0); \
		miSetGCAttrib(vis, pGC, MI_GC_LINE_WIDTH, random() % 50);
	
#define TEST_FINISH \
		miCopyPaintedSetToVisual(vis, pGC, paintedSet, offset); \
		miClearPaintedSet(vis, paintedSet); \
		miDeleteGC(vis, pGC); \
		if (ggiKbhit(vis)) break; \
	}	\
	printf("%i drawn\n", i);
	
	/* Now we run randomized drawing tests for all the XMI primitives */

	TEST_SETUP("Points");
	miDrawPoints(vis, paintedSet, pGC, MI_COORD_MODE_ORIGIN, numpoints, points);
//	miDrawPoints(vis, pGC, MI_COORD_MODE_ORIGIN, numpoints, points);
	TEST_FINISH;
	TEST_SETUP("Lines");
	miDrawLines(vis, paintedSet, pGC, MI_COORD_MODE_ORIGIN, numpoints, points);
//	miDrawLines(vis, pGC, MI_COORD_MODE_ORIGIN, numpoints, points);
	TEST_FINISH;
	TEST_SETUP("Rectangles");
	/* Define rectangle */
	rect.x = x;
	rect.y = y;   		/* upper left corner of bounding box */
	rect.width = w;           /* x range of box: 20..50 */
	rect.height = h;          /* y range of box: 15..31 */     
	miDrawRectangles(vis, paintedSet, pGC, 1, &rect);
//	miDrawRectangles(vis, pGC, 1, &rect);
	TEST_FINISH;
	TEST_SETUP("FillRectangles");
	/* Define rectangle */
	rect.x = x;
	rect.y = y;   		/* upper left corner of bounding box */
	rect.width = w;           /* x range of box: 20..50 */
	rect.height = h;          /* y range of box: 15..31 */
	miFillRectangles(vis, paintedSet, pGC, 1, &rect);
//	miFillRectangles(vis, pGC, 1, &rect);
	TEST_FINISH;
	TEST_SETUP("FillPolygon");
	miFillPolygon(vis, paintedSet, pGC, MI_SHAPE_GENERAL, MI_COORD_MODE_ORIGIN, 4, points);
//	miFillPolygon(vis, pGC, MI_SHAPE_GENERAL, MI_COORD_MODE_ORIGIN, 4, points);
	TEST_FINISH;
	TEST_SETUP("Arcs");
	/* Define elliptic arc */
	arc.x = x; 
	arc.y = y;   		/* upper left corner of bounding box */
	arc.width = w;           /* x range of box: 20..50 */
	arc.height = h;          /* y range of box: 15..31 */
	arc.angle1 = (random() % 180) * 64;      /* starting angle (1/64'ths of a degree) */
	arc.angle2 = (random() % 180) * 64;    /* angle range (1/64'ths of a degree) */          
	miDrawArcs(vis, paintedSet, pGC, 2, &arc);
//	miDrawArcs(vis, pGC, 2, &arc);
	TEST_FINISH;
	TEST_SETUP("FillArcs");
	/* Define elliptic arc */
	arc.x = x; 
	arc.y = y;   		/* upper left corner of bounding box */
	arc.width = w;           /* x range of box: 20..50 */
	arc.height = h;          /* y range of box: 15..31 */
	arc.angle1 = (random() % 180) * 64;      /* starting angle (1/64'ths of a degree) */
	arc.angle2 = (random() % 180) * 64;    /* angle range (1/64'ths of a degree) */          
	miFillArcs(vis, paintedSet, pGC, 2, &arc);
//	miFillArcs(vis, pGC, 2, &arc);
	TEST_FINISH;
	TestName("That's all folks");
	TestStart();
	while (TestTime() < 1) { };

//	miDeleteCanvas(vis, canvas);
	miDeletePaintedSet(vis, paintedSet);
	
	xmiDetach(vis);
	ggiClose(vis);
	xmiExit();	
	ggiExit();	

	return 0;
}
