/* These functions create, manipulate and destroy miGC structures.  A
 * pointer to an miGC is passed as the third argument to each of libxmi's
 * public drawing functions.  It comprises high-level drawing parameters.
 * The miGC structure is defined in mi_gc.h. */

//#include "stubsxmi.h"
//#include "mi_spans.h"
//#include "mi_gc.h"
//#include "mi_api.h"
#include "xmi_int.h"
#include "xmi_debug.h"

/* create a new miGC, with elements initialized to default values
 * (the same default values that are used by X11) */
miGC *miNewGC(ggi_visual_t vis, int npixels, const miPixel *pixels, int nblendstages, const miBlendStage *blendstages)
{
	miGC *new_gc;
	int i;
	
	XMIDPRINT_CORE("miNewGC()\n");
  
	new_gc = (miGC *)malloc(sizeof (miGC));
	new_gc->fillRule = MI_EVEN_ODD_RULE;
	new_gc->joinStyle = MI_JOIN_MITER;
	new_gc->capStyle = MI_CAP_BUTT;
	new_gc->lineStyle = MI_LINE_SOLID;
	new_gc->arcMode = MI_ARC_PIE_SLICE;
	new_gc->lineWidth = (unsigned int)0;
	new_gc->miterLimit = 10.43;	/* same as hardcoded in X11 */
	new_gc->dashOffset = 0;
	
	new_gc->numInDashList = 2;
	new_gc->dash = (unsigned int *)malloc(2 * sizeof(unsigned int));
	for (i = 0; i < 2; i++)
	  new_gc->dash[i] = 4;	/* { 4, 4 }; same as in X11? */
	
	new_gc->numPixels = npixels;
	new_gc->pixels = (miPixel *)malloc(npixels * sizeof (miPixel));
	for (i = 0; i < npixels; i++)
	  new_gc->pixels[i] = pixels[i];
	
	new_gc->numInBlendStageList = nblendstages;
	new_gc->blendstages = (miBlendStage *)malloc(nblendstages * sizeof(miBlendStage));
	for (i = 0; i < nblendstages; i++)
	  new_gc->blendstages[i] = blendstages[i];
  
	return new_gc;
}

/* Destroy (deallocate) an miGC */
void miDeleteGC(ggi_visual_t vis, miGC *pGC)
{
	XMIDPRINT_CORE("miDeleteGC()\n");
  
	if (pGC == (miGC *)NULL)
	  return;
	
	if (pGC->dash)
	  free(pGC->dash);
	
	free(pGC->pixels);
	free(pGC);
}

/* Copy an miGC */
miGC *miCopyGC(ggi_visual_t vis, const miGC *pGC)
{
	miGC *new_gc;
	int i;
  
	XMIDPRINT_CORE("miCopyGC()\n");
  
	if (pGC == (const miGC *)pGC)
	  return (miGC *)NULL;

	new_gc = (miGC *)malloc(sizeof (miGC));
	new_gc->fillRule = pGC->fillRule;
	new_gc->joinStyle = pGC->joinStyle;
	new_gc->capStyle = pGC->capStyle;
	new_gc->lineStyle = pGC->lineStyle;
	new_gc->arcMode = pGC->arcMode;
	new_gc->lineWidth = pGC->lineWidth;
	new_gc->miterLimit = pGC->miterLimit;
	new_gc->dashOffset = pGC->dashOffset;
	new_gc->numInDashList = pGC->numInDashList;
	
	if (pGC->numInDashList == 0)
	  new_gc->dash = NULL;
	else
	{
		new_gc->dash = (unsigned int *)malloc(pGC->numInDashList * sizeof(unsigned int));
		for (i = 0; i < pGC->numInDashList; i++)
		  new_gc->dash[i] = pGC->dash[i];
	}
	new_gc->pixels = (miPixel *)malloc(pGC->numPixels * sizeof(miPixel));
	for (i = 0; i < pGC->numPixels; i++)
	  new_gc->pixels[i] = pGC->pixels[i];

	return new_gc;
}

/* Set a single integer-valued miGC attribute */
void miSetGCAttrib(ggi_visual_t vis, miGC *pGC, miGCAttribute attribute, int value)
{
	XMIDPRINT_CORE("miSetGCAttrib()\n");
  
	if (pGC == (miGC *)NULL || value < 0)
	  return;
	switch (attribute)
	{
	    case MI_GC_FILL_RULE:
		pGC->fillRule = value;
		break;
	    case MI_GC_JOIN_STYLE:
		pGC->joinStyle = value;
		break;
	    case MI_GC_CAP_STYLE:
		pGC->capStyle = value;
		break;
	    case MI_GC_LINE_STYLE:
		pGC->lineStyle = value;
		break;
	    case MI_GC_ARC_MODE:
		pGC->arcMode = value;
		break;
	    case MI_GC_LINE_WIDTH:
		if (value >= 0)
		  pGC->lineWidth = (unsigned int)value;
		break;
	    default:   		/* Unknown attribute type */
		break;
	}
}

/* Set many integer-valued miGC attributes, at a single time */
void miSetGCAttribs(ggi_visual_t vis, miGC *pGC, int nattributes, const miGCAttribute *attributes, const int *values)
{
	int i;
	miGCAttribute attribute;
	int value;
  
	XMIDPRINT_CORE("miSetGCAttribs()\n");
  
	if (nattributes <= 0 || pGC == (miGC *)NULL)
	  return;
	for (i = 0; i < nattributes; i++)
	{
		attribute = *attributes++;
		value = *values++;
		
		if (value < 0)		/* invalid; be tolerant */
		  continue;
		switch ((int)attribute)
		{
		    case MI_GC_FILL_RULE:
			pGC->fillRule = value;
			break;
		    case MI_GC_JOIN_STYLE:
			pGC->joinStyle = value;
			break;
		    case MI_GC_CAP_STYLE:
			pGC->capStyle = value;
			break;
		    case MI_GC_LINE_STYLE:
			pGC->lineStyle = value;
			break;
		    case MI_GC_ARC_MODE:
			pGC->arcMode = value;
			break;
		    case MI_GC_LINE_WIDTH:
			if (value >= 0)
			  pGC->lineWidth = (unsigned int)value;
			break;
		    default:   		/* Unknown attribute type */
			break;
		}
	}
}

/* Set the only float-valued miGC attribute (the miter limit) */
void miSetGCMiterLimit(ggi_visual_t vis, miGC *pGC, double value)
{
	XMIDPRINT_CORE("miSetGCMiterLimit()\n");
  
	if (pGC == (miGC *)NULL)
	  return;
	pGC->miterLimit = value;
}

/* Set the dash-related attributes in an miGC */
void miSetGCDashes(ggi_visual_t vis, miGC *pGC, int ndashes, const unsigned int *dashes, int offset)
{
	int i;

	XMIDPRINT_CORE("miSetGCDashes()\n");
  
	if (pGC == (miGC *)NULL || ndashes < 0)
	  return;
	if (pGC->dash)
	  free(pGC->dash);
	pGC->dashOffset = offset;
	pGC->numInDashList = ndashes;
	if (ndashes == 0)
	  pGC->dash = (unsigned int *)NULL;
	else
	{
		pGC->dash = (unsigned int *)malloc(ndashes * sizeof(unsigned int));
		for (i = 0; i < ndashes; i++)
		  pGC->dash[i] = dashes[i];
	}
}

/* Set the pixel array in a miGC */
void miSetGCPixels(ggi_visual_t vis, miGC *pGC, int npixels, const miPixel *pixels)
{
	int i;

	XMIDPRINT_CORE("miSetGCPixels()\n");
  
	if (pGC == (miGC *)NULL || npixels < 2)
	  return;

	pGC->numPixels = npixels;
	pGC->pixels = (miPixel *)realloc(pGC->pixels, npixels * sizeof (miPixel));

	for (i = 0; i < npixels; i++)
	  pGC->pixels[i] = pixels[i];
}
