/******************************************************************************
 * 
 * 
 * Master LibXMI headerfile
 * 
 * 
 *****************************************************************************
 */

#ifndef _GGI_XMI_H
#define _GGI_XMI_H

#include <ggi/ggi.h>


/* This version of xmi.h is appropriate for GGI libxmi version 1.1. */
#define LIBXMI_VERSION "1.1"

/* support C++ */
#ifdef ___BEGIN_DECLS
#undef ___BEGIN_DECLS
#endif
#ifdef ___END_DECLS
#undef ___END_DECLS
#endif
#ifdef __cplusplus
# define ___BEGIN_DECLS extern "C" {
# define ___END_DECLS }
#else
# define ___BEGIN_DECLS		/* empty */
# define ___END_DECLS		/* empty */
#endif


/* Structure that defines a point with integer coordinates. */
typedef struct {
	int x, y;			/* integer coordinates, y goes downward */
} miPoint;

/* FIXME: miPixel is always a ggi_pixel */
typedef ggi_pixel miPixel;

/* A Spans structure is a sorted list of spans, i.e. a list of point ranges
 * [xmin,xmax], sorted in increasing y order.  There may be more than one
 * span at a given y. */
typedef struct 
{
	unsigned int count;    	/* Number of spans		    */
	miPoint *points;	/* Pointer to list of start points  */
	unsigned int *widths;	/* Pointer to list of widths	    */
} Spans;

/* A SpanGroup is an unsorted list of Spans's, associated with a particular
 * pixel value.  
 *
 * A SpanGroup is allowed to include more than a single Spans because most
 * libxmi drawing functions write out multiple Spans's. */
typedef struct {
	miPixel	pixel;		/* pixel value				*/
	Spans   *group;		/* Spans slots				*/
	unsigned int size;     	/* Number of Spans slots allocated	*/
	unsigned int count;    	/* Number of Spans slots filled		*/
	int ymin, ymax;		/* min, max y values over all Spans's	*/
} SpanGroup;

/* A miPaintedSet structure is an array of SpanGroups, specifying the
 * partition into differently painted subsets.  There is at most one
 * SpanGroup for any pixel. */
typedef struct lib_miPaintedSet {
	SpanGroup **groups;	/* SpanGroup slots			*/
	unsigned int size;     	/* Number of SpanGroup slots allocated	*/
	unsigned int ngroups;	/* Number of SpanGroup slots filled	*/
} miPaintedSet;

/* libxmi's low-level painting macro.  It `paints' a Spans, i.e. a list of
 * spans assumed to be in y-increasing order, to a miPaintedSet with a
 * specified pixel value.  To do this, it invokes the lower-level function
 * miAddSpansToPaintedSet() in mi_spans.c.
 * 
 * The passed point and width arrays should have been allocated on the
 * heap, since they will be eventually freed; e.g., when the miPaintedSet
 * is cleared or deallocated. */

#define MI_PAINT_SPANS(paintedSet, pixel, numSpans, ppts, pwidths) \
  {\
    Spans 	spanRec;\
    if (numSpans > 0) \
      { \
        spanRec.points = (ppts);\
        spanRec.widths = (pwidths);\
        spanRec.count = (numSpans);\
        XMI_stubs_addspanstopaintedset(vis, &spanRec, (paintedSet), (pixel));\
      } \
    else \
      { \
        free (ppts); \
        free (pwidths); \
      } \
    }

/* A wrapper for MI_PAINT_SPANS() that can be applied to a span array
 * (i.e. to a point and width array) that can't be freed, so must be
 * copied.  We try not to use this. */
#define MI_COPY_AND_PAINT_SPANS(paintedSet, pixel, nPts, FirstPoint, FirstWidth) \
  {\
    if ((nPts) > 0) \
      { \
        miPoint		*ppt, *pptInit, *oldppt; \
        unsigned int	*pwidth, *pwidthInit, *oldpwidth; \
        int		ptsCounter; \
        ppt = pptInit = (miPoint *) malloc ((nPts) * sizeof (miPoint));\
        pwidth = pwidthInit = (unsigned int *) malloc ((nPts) * sizeof (unsigned int));\
        oldppt = FirstPoint;\
        oldpwidth = FirstWidth;\
        for (ptsCounter = (nPts); --ptsCounter >= 0; )\
          {\
            *ppt++ = *oldppt++;\
            *pwidth++ = *oldpwidth++;\
          }\
        MI_PAINT_SPANS(paintedSet, pixel, (nPts), pptInit, pwidthInit)\
      } \
  }

___BEGIN_DECLS

/* miPaintedSet manipulation routines (other than public) */
extern void XMI_stubs_addspanstopaintedset(ggi_visual_t vis, Spans *spans, miPaintedSet *paintedSet, miPixel pixel);
extern void XMI_stubs_quicksortspansy(ggi_visual_t vis, miPoint *points, unsigned int *widths, int numSpans);
extern void XMI_stubs_uniquifypaintedset(ggi_visual_t vis, miPaintedSet *paintedSet);

extern miPaintedSet *miNewPaintedSet(ggi_visual_t vis);
extern void miDeletePaintedSet(ggi_visual_t vis, miPaintedSet *paintedSet);
extern void miClearPaintedSet(ggi_visual_t vis, miPaintedSet *paintedSet);

___END_DECLS

/* Blend stage
 * 
 * This is a generic three-operand pixel blending stage list element,
 * with three source operands and a abstract stage type.
 */
typedef struct lib_miBlendStage {
	int type;		/* Blend stage type */
	void *op1, *op2;	/* Untyped operand pointers */
} miBlendStage;

/* Blend stage types
 * 
 * We abstract these so that they can be efficiently accelerated by the
 * targets, singly or in groups, per-pixel or per-drawingop. */
typedef enum {
	MI_BLEND_NULL,			/* Destination == source	*/
	
	MI_BLEND_COLORKEY,		/* Draw iff (color != OP1)	*/
					/* OP1 = key color (ggi_pixel) 	*/
					/* OP2 = Unused		       	*/
	
	MI_BLEND_ROP,			/* Standard logical rasterop   	*/
					/* OP1 = ROP code 	       	*/
					/* OP2 = Unused		       	*/
	
	MI_BLEND_TEXTURE_ROP,		/* ROP-enabled texture mapping 	*/
					/* OP1 = ROP code 	       	*/ 
					/* OP2 = Texture pixmap	       	*/

	MI_BLEND_TEXTURE_FILTER,       	/* Texture filtering	       	*/
					/* OP1 = Filter type 	       	*/ 
					/* OP2 = Texture pixmap        	*/
	
	MI_BLEND_TEXTURE_MULTI,       	/* Multitexture blend	       	*/
					/* OP1 = Texture pixmap array  	*/ 
					/* OP2 = Texture blend op      	*/
	
	MI_BLEND_ALPHA_BUFFER,  	/* Alpha buffer blend	       	*/
					/* OP1 = Alpha blend operation 	*/
					/* OP2 = Alpha buffer pixmap   	*/
	
	MI_BLEND_ALPHA_CONSTANT,  	/* Alpha constant blend	       	*/
					/* OP1 = Alpha blend operation 	*/
					/* OP2 = Alpha constant        	*/
	
	MI_BLEND_ALPHA_TEST,   		/* Alpha test blend    	       	*/
					/* OP1 = Alpha test operation  	*/
					/* OP2 = Alpha buffer pixmap   	*/
	
	MI_BLEND_STENCIL,	       	/* Stencil mask test	       	*/
					/* OP1 = Stencil pixmap        	*/
					/* OP2 = Unused		       	*/
	  
	MI_BLEND_Z		       	/* Z-buffer operations         	*/
					/* OP1 = Z-buffer pixmap       	*/
					/* OP2 = Z op parameter struct	*/
	  
} miBlendType;

typedef enum {
	ROP_PAINT
} miRopType;

/* Z-buffer test/update operations */
typedef enum {
	/* Write type */
	Z_WRITE_UPDATE_DEST,   		/* Update destination pixel Y/N */
	Z_WRITE_UPDATE_ZVAL,		/* Update Z-buffer value Y/N	*/
	Z_WRITE_ZTEST,			/* Perform Z-test Y/N 		*/
	
	Z_WRITE_TEST_EQ = 0x10000000,	/* (Dc = Sc) iff Sz == Dz      	*/
	Z_WRITE_TEST_NE = 0x20000000,	/* (Dc = Sc) iff Sz != Dz      	*/
	Z_WRITE_TEST_GT = 0x30000000,	/* (Dc = Sc) iff Sz > Dz      	*/
	Z_WRITE_TEST_GE = 0x40000000,	/* (Dc = Sc) iff Sz >= Dz      	*/
	Z_WRITE_TEST_LT = 0x50000000,	/* (Dc = Sc) iff Sz < Dz      	*/
	Z_WRITE_TEST_LE = 0x60000000	/* (Dc = Sc) iff Sz <= Dz      	*/
} miZBlendType;

typedef struct {
	miZBlendType zBlendType;      	
	int zval;			/* Constant Z value		*/
} miZBlendOpDataType;


/* Blend stage parameter types:
 * 
 * sc = source color (32 bits)
 * dc = destination color (32 bits)
 * sa = source alpha value (8 bits)
 * da = destination alpha value (8 bits)
 * ac = alpha constant (8 bits)
 * ab = alpha buffer value (8 bits)
 * stb = stencil buffer value (1 bit)
 */

/* Alpha blending types */
typedef enum {
	BLEND_SC,
	BLEND_DC,
	BLEND_SC_OR_DC,		/* dc = (sc | dc)      	*/
	BLEND_SC_AND_DC,       	/* dc = (sc & dc)      	*/
	BLEND_SC_AND_AC,      	/* dc = (sc & ac)      	*/
	BLEND_SC_AND_AB,      	/* dc = (sc & ab)     	*/
	BLEND_SC_AND_AC_OR_DC, 	/* dc = (sc & ac) | dc	*/
	BLEND_SC_AND_AB_OR_DC, 	/* dc = (sc & ab) | dc  */
	BLEND_SC_AND_STB,	/* dc = (dc & stb)	*/
	BLEND_S_MINUS_D		/* dc = (sc - dc)      	*/
} miAlphaBlendType;

/* Alpha testing types */
typedef enum {
	TEST_S_EQ_D,		/* (Dc = Sc) iff Sa == Da      	*/
	TEST_S_NE_D,		/* (Dc = Sc) iff Sa != Da      	*/
	TEST_S_GT_D,		/* (Dc = Sc) iff Sa > Da       	*/
	TEST_S_GE_D,		/* (Dc = Sc) iff Sa >= Da      	*/
	TEST_S_LT_D,		/* (Dc = Sc) iff Sa < Da       	*/
	TEST_S_LE_D		/* (Dc = Sc) iff Sa <= Da      	*/
} miAlphaTestType;

/* Texture filtering types */
typedef enum {
	FILTER_REPLICATE,      	/* Duplicate or remove pixels  	*/
	FILTER_BILINEAR,	/* Bilinear filter	       	*/
	FILTER_TRILINEAR,	/* Trilinear filter	       	*/
	FILTER_ANISOTROPIC,	/* Anisotropic filter	       	*/
} miTextureFilterType;


/* A miGC is an opaque structure that contains high-level drawing parameters.
 * We draw 'through' an miGC struct onto a ggi_visual.  When any public 
 * XMI drawing function is invoked, a pointer to an miGC is always passed as
 * its second argument. */
typedef struct {
	/* High-level drawing parameters not related to dashing */
	int fillRule;	       	/* Default = miEvenOddRule */
	int joinStyle;	       	/* Default = miJoinMiter */
	int capStyle;	       	/* Default = miCapButt */
	int lineStyle;	       	/* Default = miLineSolid */
	int arcMode;	       	/* Default = miArcPieSlice */
	unsigned int lineWidth;	/* Line thickness in pixels (default = 0) */
	double miterLimit;     	/* Default = 10.43, as in X11 */

	/* Array of pixel types, used for dashing */
	miPixel *pixels;       	/* Array of pixel types */
	int numPixels;	       	/* Number of pixel types (must be >=2) */
	
	/* High-level drawing parameters related to dashing */
	unsigned int *dash;    	/* Dash array (lengths of dashes in pixels) */
	int numInDashList;     	/* Length of dash array */
	int dashOffset;	       	/* Pixel offset of first dash (nonnegative) */

	/* Blend stages */
	miBlendStage *blendstages;  	/* Head of blend stage list */
	int numInBlendStageList;	/* Length of blend stage array */
} miGC;

___BEGIN_DECLS

/* Constructor, destructor, and copy constructor for the miGC class */
extern miGC *miNewGC(ggi_visual_t vis, int npixels, const miPixel *pixels, int nblendstages, const miBlendStage *blendstages); 
extern void miDeleteGC(ggi_visual_t vis, miGC *pGC);
extern miGC *miCopyGC(ggi_visual_t vis, const miGC *pGC);

___END_DECLS

/* Values for an miGC's miGCFillRule attribute (default=MI_EVEN_ODD_RULE) */
enum { MI_EVEN_ODD_RULE, MI_WINDING_RULE };

/* Values for an miGC's miGCJoinStyle attribute, default == MI_JOIN_MITER */
enum { MI_JOIN_MITER, MI_JOIN_ROUND, MI_JOIN_BEVEL, MI_JOIN_TRIANGULAR };

/* Values for an miGC's miGCCapStyle attribute (default=MI_CAP_BUTT).
 * MI_CAP_NOT_LAST is the same as MI_CAP_BUTT except when drawing
 * zero-width (Bresenham) polylines; it causes the final pixel not to be
 * drawn.  A polyline drawn in this way is called `continuable'. */
enum { MI_CAP_NOT_LAST, MI_CAP_BUTT, MI_CAP_ROUND, MI_CAP_PROJECTING, MI_CAP_TRIANGULAR };

/* Values for an miGC's miGCLineStyle attribute, default == MI_LINE_SOLID */
enum { MI_LINE_SOLID, MI_LINE_ON_OFF_DASH, MI_LINE_DOUBLE_DASH };

/* Values for an miGC's miGCArcMode attribute, default == MI_ARC_PIE_SLICE */
enum { MI_ARC_CHORD, MI_ARC_PIE_SLICE };

/* Possibilities for the `attribute' argument of miSetGCAttrib.  All the
 * preceding, plus MI_GC_LINE_WIDTH */
typedef enum { MI_GC_FILL_RULE, MI_GC_JOIN_STYLE, MI_GC_CAP_STYLE, MI_GC_LINE_STYLE, MI_GC_ARC_MODE, MI_GC_LINE_WIDTH } miGCAttribute;

___BEGIN_DECLS

/* A function that sets a single integer-valued miGC attribute.  `value'
 * must be one of the preceding enum's, except when
 * attribute = MI_GC_LINE_WIDTH, in which case value >=0 is required. */
extern void miSetGCAttrib(ggi_visual_t vis, miGC *pGC, miGCAttribute attribute, int value);

/* A function that sets a list of integer-value miGC attributes. */
extern void miSetGCAttribs(ggi_visual_t vis, miGC *pGC, int nattributes, const miGCAttribute *attributes, const int *values);

/* Functions that set miGC attributes that are not integer-valued.
 * Note: currently, `offset' must be nonnegative. */
extern void miSetGCDashes(ggi_visual_t vis, miGC *pGC, int ndashes, const unsigned int *dashes, int offset);
extern void miSetGCMiterLimit(ggi_visual_t vis, miGC *pGC, double miter_limit);
extern void miSetGCPixels(ggi_visual_t vis, miGC *pGC, int npixels, const miPixel *pixels);
extern void miSetGCBlendStages(ggi_visual_t vis, miGC *pGC, int nblendstages, const miBlendStage *blendstages);

___END_DECLS

/*********** DECLARATIONS OF PUBLIC DRAWING FUNCTIONS ******************/

/* The semantics of these drawing functions is similar to that of the
 * corresponding X11 drawing functions.  Wide polylines (polylines with
 * line width >= 1) are treated as polygons to be rendered by filling.
 * Zero-width polylines are not invisible: instead, they are single-pixel
 * polylines, specially rendered by the Bresenham midpoint line algorithm.
 * 
 * Also, adjoining polygons (those with an edge in common) are drawn
 * without gaps.  To arrange this, the `right' and `bottom' edges of any
 * polygon are not drawn when the polygon is filled.  The filling of
 * rectangles is similar.
 * 
 * Wide arcs and polyarcs are drawn with a circular brush, of diameter
 * equal to the line width.  Every brushed pixel is painted.  Zero-width
 * arcs and polyarcs are not invisible: instead, they are single-pixel
 * arcs, specially rendered by the Bresenham `midpoint arc' algorithm. */

/* For consistency, the first two arguments of each drawing function are
 * (1) a pointer to a ggi_visual, and (2) a pointer to an miGC. */

/* 1. Drawing functions for points, polylines, and polygons.
 * 
 * The final three arguments of each are a `coordinate mode' (see below), a
 * specified number of points, and an array that contains the points.
 * miDrawPoints draws a cloud of points, miDrawLines draws a polyline, and
 * miFillPolygon draws a filled polygon. */

/* Possible values for the `coordinate mode' argument specifying whether
 * the points in the points array, after the first point, are given in
 * absolute or relative coordinates. */
typedef enum { MI_COORD_MODE_ORIGIN, MI_COORD_MODE_PREVIOUS } miCoordMode;

/* Possible values for the `shape' argument of miFillPolygon().  Two
 * possibilities: (1) general (i.e., not necessarily convex, with
 * self-intersections allowed), or (2) convex and not self-intersecting.
 * Latter case can be drawn more rapidly. */
typedef enum { MI_SHAPE_GENERAL, MI_SHAPE_CONVEX } miPolygonShape;

___BEGIN_DECLS

extern void miDrawPoints(ggi_visual_t vis, miPaintedSet *paintedSet, const miGC *pGC, miCoordMode mode, int npts, const miPoint *pPts);
extern void miDrawLines(ggi_visual_t vis, miPaintedSet *paintedSet, const miGC *pGC, miCoordMode mode, int npts, const miPoint *pPts);
extern void miFillPolygon(ggi_visual_t vis, miPaintedSet *paintedSet, const miGC *pGC, miPolygonShape shape, miCoordMode mode, int npts, const miPoint *pPts);

___END_DECLS

/* Structure that defines a rectangle.  Upper left corner is [x,y] and
 * lower right corner is [x+width,y+height]. */
typedef struct {
	int x, y;			/* Upper left corner */
	unsigned int width, height;	/* width >= 1 and height >= 1 */
} miRectangle;

___BEGIN_DECLS

/* 2. Rectangle-related drawing functions. These draw and fill a specified 
 * number of rectangles, supplied as an array of miRectangles. */
extern void miDrawRectangles(ggi_visual_t vis, miPaintedSet *paintedSet, const miGC *pGC, int nrects, const miRectangle *pRects);
extern void miFillRectangles(ggi_visual_t vis, miPaintedSet *paintedSet, const miGC *pGC, int nrects, const miRectangle *pRects);

___END_DECLS

/* Structure that defines an `arc' (i.e. a segment of an ellipse whose
 * principal axes are aligned with the coordinate axes).  The upper left
 * corner of the bounding box is [x,y], and the lower right corner is
 * [x+width,y+height].  By convention, angle1 and angle2 are the starting
 * polar angle and angle range that the arc would have if it were scaled
 * into a circular arc. */
typedef struct {
	int x, y;			/* Upper left corner of ellipse's bounding box */
	unsigned int width, height;	/* Dimensions; width, height >= 1 */
	int angle1, angle2;		/* Starting angle and angle range, in 1/64 degrees */
} miArc;

/* 3. Arc-related drawing functions.
 * 
 * Each of these takes as argument a multi-arc, i.e. an array of elliptic
 * arcs.  Here, an `elliptic arc' is a piece of an ellipse whose axes are
 * aligned with the coordinate axes.  The arcs are not required to be
 * contiguous.
 * 
 * miDrawArcs draws a multi-arc.  If the arcs are contiguous, they will be
 * joined as specified in the miGC.  Note that miDrawArcs is not reentrant,
 * i.e., not thread-safe (for a thread-safe variant, see below).
 *
 * miFillArcs draws a sequence of filled arcs.  They are filled either as
 * chords or as pie slices, as specified by the graphics context. */

___BEGIN_DECLS

extern void miDrawArcs(ggi_visual_t vis, miPaintedSet *paintedSet, const miGC *pGC, int narcs, const miArc *parcs);
extern void miFillArcs(ggi_visual_t vis, miPaintedSet *paintedSet, const miGC *pGC, int narcs, const miArc *parcs);

___END_DECLS

/* miPixmap
 * 
 * This is a generic untyped sub-bitmap representation, used as a source or
 * destination operator for an miBlendStage.
 * Each one is tied to a ggi_visual.*/
typedef struct {
	uint flags;
	ggi_visual_t vis;
	miRectangle rect;
} miPixmap;

/* Pixmap flags */
typedef enum {
	PIXMAP_DB,		/* DirectBuffer	is available	*/
	PIXMAP_VIS		/* ggi_visual is available	*/
} miPixmapFlags;

___BEGIN_DECLS

extern miPixmap *miNewPixmap(uint w, uint h, ggi_graphtype type);
extern void miDeletePixmap(ggi_visual_t vis, miPixmap *pPixmap);

extern void miCopyPaintedSetToVisual(ggi_visual_t vis, const miGC *pGC, const miPaintedSet *paintedSet, miPoint offset);

/* Extension management */
int xmiInit(void);
int xmiExit(void);

int xmiAttach(ggi_visual_t vis);
int xmiDetach(ggi_visual_t vis);

___END_DECLS

#endif /* _GGI_XMI_H */
