/*
 ******************************************************************************

 LIBGPF: dynamic loading

 Copyright (C) 2000-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */


#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_internal.h>
#include <ggi/internal/gpf_helpers.h>
#include <ggi/internal/gpf_dl.h>
#include <ggi/internal/gpf_debug.h>

#include <string.h>


#define GPF_SYMNAME_MAX         255
#define GPF_SYMNAME_PREFIX      "GPFdl_"


/* Open the dynamic libary requested
 */
static int _gpfLoadDL(const char *filename, const char *symprefix,
			int type, gpf_dlhandle **dlh)
{
	gpf_dlhandle hand;
	char symname[GPF_SYMNAME_MAX+1], *nameptr;

	GPFDPRINT_LIBS("_gpfLoadDL(\"%s\", 0x%x) called \n", filename);

	hand.name = NULL;
	hand.usecnt = 0;

	if (type & GPF_DLTYPE_GLOBAL) {
		hand.handle = ggLoadModule(filename, GG_MODULE_GLOBAL);
	} else {
		hand.handle = ggLoadModule(filename, 0);
	}	/* if */

	GPFDPRINT_LIBS("hand.handle=%p\n", hand.handle);
	if (hand.handle == NULL) {
		GPFDPRINT_LIBS("Error loading module %s\n", filename);
		return GGI_ENOFILE;
	}	/* if */

	nameptr = strrchr(filename, '/');
	if (!nameptr) {
		nameptr = (char*)filename;
	} else {
		nameptr++;
	}	/* if */

	sprintf(symname, "%s%s", symprefix, nameptr);
	nameptr = strrchr(symname, '.');
	if (nameptr) {
		*nameptr = '\0';
	}	/* if */

	hand.entry = ggGetSymbolAddress(hand.handle, symname);
	GPFDPRINT_LIBS("&(%s) = %p\n", symname, hand.entry);
	if (hand.entry == NULL) {
		ggFreeModule(hand.handle);
		return GGI_ENOFUNC;
	}	/* if */

	/* Get pointers */
	hand.entry(GPFFUNC_open, (void**)&hand.open);
	hand.entry(GPFFUNC_exit, (void**)&hand.exit);
	hand.entry(GPFFUNC_close, (void**)&hand.close);
	GPFDPRINT_LIBS("hand.open = %p\n", hand.open);
	GPFDPRINT_LIBS("hand.exit = %p\n", hand.exit);
	GPFDPRINT_LIBS("hand.close = %p\n", hand.close);

	*dlh = malloc(sizeof(**dlh));
	if (*dlh == NULL) {
		ggFreeModule(hand.handle);
		return GGI_ENOMEM;
	}	/* if */
	memcpy(*dlh, &hand, sizeof(gpf_dlhandle));

	return GGI_OK;
}	/* _gpfLoadDL */


int _gpfAddDL(struct gpf_handle_opio_t *opio, struct gpf_pipeline_item_t *pipe,
		const char *name, const char *args,
		va_list ap, int type)
{
	gpf_dlhandle_l *tmp;
	gpf_dlhandle *dlh;
	const char *filename;
	uint32 dlret = GGI_OK;
	int err;

	GPFDPRINT_LIBS("_gpfAddDL(%p, \"%s\", \"%s\", 0x%x) called\n",
			pipe, name, args ? args : "(null)", type);

	if ((filename = ggMatchConfig(_gpfConfigHandle, name, NULL)) == NULL) {
		GPFDPRINT_LIBS("no config entry for sublib: %s\n",
				name);
		return GGI_ENOMATCH;
	}	/* if */

	err = _gpfLoadDL(filename, GPF_SYMNAME_PREFIX, type, &dlh);
	GPFDPRINT_LIBS("_gpfLoadDL returned %d (%p)\n", err, dlh);
	if (err) return err;

	dlh->pipe = pipe;
	if (opio != NULL) {
		/* happens only, when opening a target/protocol sublib */
		pipe->istarget = 1;
		opio->type = type;
	}	/* if */

	err = dlh->open(opio, pipe, dlh, args, ap, &dlret);
	GPFDPRINT_LIBS("%d = dlh->open(%p, %p, %p, \"%s\", %p, %d) - %s\n",
			err, opio, pipe, dlh, args ? args : "(null)",
			ap, dlret, filename);
	if (err) {
		ggFreeModule(dlh->handle);
		free(dlh);
		return err;
	}	/* if */


	tmp = (gpf_dlhandle_l *)_gpf_malloc(sizeof(gpf_dlhandle_l));
	tmp->handle = dlh;

	LIB_ASSERT(pipe != NULL);
	tmp->next = pipe->head.dlhandle;
	pipe->head.dlhandle = tmp;


	dlh->usecnt++;


	if (dlh->usecnt == 0) {
		fprintf(stderr,
			"LibGPF: %s (%s) -> 0x%.8x - no operations in this library\n",
			name, args ? args : "(null)", dlret);
		ggFreeModule(dlh->handle);
		free(dlh);
		return GGI_ENOFUNC;
	} else {
		tmp = (gpf_dlhandle_l *)_gpf_malloc(sizeof(gpf_dlhandle_l));
		tmp->handle = dlh;

		tmp->next = LIBGPF_PIPE_DLHANDLE(pipe);
		LIBGPF_PIPE_DLHANDLE(pipe) = tmp;
	}	/* if */

	dlh->name = strdup(name);

	return GGI_OK;
}	/* _gpfAddDL */



int _gpfOpenDL(struct gpf_handle_opio_t *opio, struct gpf_pipeline_item_t *pipe,
		const char *name, const char *args, va_list ap, int type)
{
	return _gpfAddDL(opio, pipe, name, args, ap, type);
}	/* _gpfOpenDL */


void _gpfExitDL(struct gpf_pipeline_item_t *pipe,
		gpf_dlhandle_l *lib)
{
	for (; lib != NULL; lib = lib->next) {
		if (lib->handle->exit) {
			lib->handle->exit(pipe, lib->handle);
		}	/* if */
	}	/* for */
}	/* _gpfExitDL */


static void _gpfRemoveDL(struct gpf_pipeline_item_t *pipe,
			gpf_dlhandle_l **lib)
{
	gpf_dlhandle_l *tmp = NULL, **prev = NULL;
	gpf_dlhandle_l *libtmp, **libprev, *libnext;

	for (libprev = lib, libtmp = *lib; libtmp != NULL; libtmp=libnext) {
		libnext = libtmp->next;
		if (libtmp->handle->usecnt <= 0) {

			GPFDPRINT_LIBS("Disposing \"%s\"\n",
					libtmp->handle->name);
			*libprev = libtmp->next;
			if (libtmp->handle->close) {
				libtmp->handle->close(pipe, libtmp->handle);
			}	/* if */

			GPFDPRINT_LIBS("Closing handle: 0x%x\n",
					libtmp->handle->handle);
			ggFreeModule(libtmp->handle->handle);

			/* Now, clean up the master handle */

			prev = &LIBGPF_PIPE_DLHANDLE(pipe);
			tmp = LIBGPF_PIPE_DLHANDLE(pipe);

			for (; tmp; tmp=tmp->next) {
				if (tmp->handle == libtmp->handle) break;
				prev = &tmp->next;
			}	/* for */
			if (!tmp) {
				GPFDPRINT_LIBS("Error: handle not in master list.\n");
			}	/* if */
			*prev = tmp->next;
			free(tmp);

			free(libtmp->handle->name);
			free(libtmp->handle);
			free(libtmp);
		} else {
			libprev = &libtmp->next;
		}	/* if */
	}	/* for */
}	/* _gpfRemoveDL */


void _gpfZapDL(struct gpf_pipeline_item_t *pipe, gpf_dlhandle_l **lib)
{
	gpf_dlhandle_l *tmp, *next;

	GPFDPRINT_LIBS("_gpfZapDL(%p, %p) called\n", pipe, lib);

	for (tmp = *lib; tmp; tmp = tmp->next) {
		tmp->handle->usecnt--;
	}	/* for */

	_gpfRemoveDL(pipe, lib);

	for (tmp = *lib; tmp; tmp = next) {
		next = tmp->next;
		free(tmp);
	}	/* for */

	*lib = NULL;
}	/* _gpfZapDL */




int _gpfCloseDL(struct gpf_pipeline_item_t *pipe)
{
	_gpfExitDL(pipe, pipe->head.dlhandle);
	_gpfZapDL(pipe, &pipe->head.dlhandle);

	return GGI_OK;
}	/* _gpfCloseDL */
