/*
 ******************************************************************************

 LIBGPF: plain functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_debug.h>

#include <string.h>


#if 0

/* get the most important image data
 */
int gpfGetFileInfo(gpf_handle_t handle, struct gpf_fileinfo_t **info)
{
	int rc;

	if (handle == NULL) return GGI_EARGINVAL;
	if (info == NULL) return GGI_EARGINVAL;
	if (info[0] != NULL) return GGI_EARGINVAL;

	info[0] = calloc(1, sizeof(struct gpf_fileinfo_t));
	if (!info[0]) {
		/* not enough memory */
		rc = GGI_ENOMEM;
		goto err0;
	}	/* if */

	memcpy(info[0], &(handle->fileinfo), sizeof(struct gpf_fileinfo_t));
	LIB_ASSERT(info[0]->datasize == handle->fileinfo.datasize);

	if (info[0]->datasize != 0) {
		info[0]->data = calloc(1, info[0]->datasize);
		if (!info[0]->data) {
			/* not enough memory */
			rc = GGI_ENOMEM;
			goto err1;
		}	/* if */

		memcpy(info[0]->data, handle->fileinfo.data, info[0]->datasize);
	}	/* if */

	return GGI_OK;

err1:
	free(info[0]);
	info[0] = NULL;
err0:
	return rc;
}	/* gpfGetFileInfo */

#endif



int gpfIsStopped(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	return (!handle->started) ? 1 : 0;
}	/* gpfIsStopped */



int gpfStartStream(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	if (gpfIsEOS(handle)) {
		GPFDPRINT("gpfStartStream: End Of Stream reached - stop transfer!\n");
		gpfStopStream(handle);
	} else {
		handle->started = 1;
		ggCurTime(&handle->tv);

		if ((handle->pipe_index != NULL)
		  && (handle->props.flags & GPF_FRAME_CONTINUE))
		{
			/* we continue here and do not restart */
			GPFDPRINT("gpfStreamStart: stream transfer continues\n");
			return GGI_OK;
		}	/* if */

		handle->pipe_lasttransfer = NULL;
		handle->pipe_index = handle->pipeline->item_head;
		GPFDPRINT("gpfStreamStart: stream transfer (re)started\n");
	}	/* if */

	return GGI_OK;
}	/* gpfStartStream */


int gpfStopStream(gpf_handle_t handle)
{
	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	handle->started = 0;
	GPFDPRINT_CORE("gpfStopStream: stream transfer stopped\n");
	return GGI_OK;
}	/* gpfStopStream */


int gpfIsEOS(gpf_handle_t handle)
{
	int is_eos;

	GPF_APPASSERT(handle != NULL, "Invalid handle! Please gpfOpen() it properly.\n");

	is_eos = handle->opio_input->opproto->eos(handle->opio_input);
	GPFDPRINT_CORE("gpfIsEOS: input stream %s\n",
			(is_eos) ? "reached End Of Stream" : "can continue");
	if (is_eos) return is_eos;

	is_eos = handle->opio_output->opproto->eos(handle->opio_output);
	GPFDPRINT_CORE("gpfIsEOS: output stream %s\n",
			(is_eos) ? "reached End Of Stream" : "can continue");
	if (is_eos) return is_eos;

	return GGI_OK;
}	/* gpfIsEOS */



enum gpf_flags gpfSetFlags(struct gpf_properties_t *props,
			enum gpf_flags flags)
{
	enum gpf_flags tmp;

	LIB_ASSERT(props != NULL);

	tmp = (props->flags & GPF_RESPONSE_MASK);
	flags &= GPF_REQUEST_MASK;
	flags |= (props->flags & ~GPF_REQUEST_MASK);

	props->flags = flags;

	return tmp;
}	/* gpfSetFlags */


enum gpf_flags gpfGetFlags(const struct gpf_properties_t *props)
{
	LIB_ASSERT(props != NULL);

	return (props->flags & GPF_RESPONSE_MASK);
}	/* gpfGetFlags */



int gpfClearProperties(struct gpf_properties_t *props)
{
	LIB_ASSERT(props != NULL);
	memset(props, GPF_AUTO, sizeof(struct gpf_properties_t));

	props->flags = GPF_FRAME_CONTINUE;
	props->flags |= GPF_PIPE_OPT4SPEED;
	props->flags |= GPF_RATE_KBYTE;

	return GGI_OK;
}	/* gpfClearProperties */


int gpfSetProperties(gpf_handle_t handle, const struct gpf_properties_t *props)
{
	int flags;
	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(props != NULL);

	if (!gpfIsStopped(handle)) {
		return GGI_EBUSY;
	}	/* if */

	flags = props->flags;
	if ((flags & GPF_FRAME_MASK) == 0) {
		flags |= (LIBGPF_FLAGS(handle) & GPF_FRAME_MASK);
	}	/* if */
	if ((flags & GPF_PIPE_MASK) == 0) {
		flags |= (LIBGPF_FLAGS(handle) & GPF_PIPE_MASK);
	}	/* if */
	if ((flags & GPF_RATE_MASK) == 0) {
		flags |= (LIBGPF_FLAGS(handle) & GPF_RATE_MASK);
	}	/* if */
	memcpy(&(LIBGPF_PROPS(handle)), props, sizeof(struct gpf_properties_t));
	LIBGPF_FLAGS(handle) = flags;

	return GGI_OK;
}	/* gpfSetProperties */


int gpfGetProperties(gpf_handle_t handle, struct gpf_properties_t *props)
{
	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(props != NULL);

	memcpy(props, &(LIBGPF_PROPS(handle)), sizeof(struct gpf_properties_t));

	return GGI_OK;
}	/* gpfGetProperties */



ssize_t gpfTransferNextStreamBlock(gpf_handle_t handle,
				gpfIndicatorCallback cbI)
{
	struct timeval newtv, diff;
	struct gpf_pipeline_item_t *pipe_tmp;
	size_t transfer_time;
	int transfer_rate;
	ssize_t transfer_size, bytes_transfered, sum_bytes_transfered;


	GPFDPRINT_TRANSFER("gpfTransferNextStreamBlock(%p, %p) called\n",
			handle, cbI);

	if (gpfIsStopped(handle)) {
		GPFDPRINT("stream transfer is stopped - abort\n");

		/* stream is stopped/hasn't been started */
		return 0;
	}	/* if */


	ggCurTime(&newtv);
	diff.tv_sec = newtv.tv_sec - handle->tv.tv_sec;
	diff.tv_usec = newtv.tv_usec - handle->tv.tv_usec;
	if (diff.tv_usec < 0) {
		diff.tv_usec += 1000000;
		diff.tv_sec--;
	}	/* if */

	transfer_time = (diff.tv_sec * 1000000 + diff.tv_usec) / 1000;
	if (transfer_time == 0) transfer_time = 1024;	/* CPU too fast? */

	transfer_rate = (LIBGPF_RATE(handle) == 0) ? 1 : LIBGPF_RATE(handle);

	switch (LIBGPF_FLAGS(handle) & GPF_RATE_MASK) {
	case GPF_RATE_KBYTE:
		transfer_rate *= (1024);
		break;

	case GPF_RATE_MBYTE:
		transfer_rate *= (1024 * 1024);
		break;

	case GPF_RATE_KBIT:
		transfer_rate *= (1024 / 8);
		break;

	case GPF_RATE_MBIT:
		transfer_rate *= (1024 * 1024 / 8);
		break;
	}	/* switch */

	transfer_size = transfer_time * transfer_rate;
	/* BTW: bytes_per_second = 1000 / transfer_time; */

	memcpy(&handle->tv, &newtv, sizeof(struct timeval));

	sum_bytes_transfered = 0;
	GPFDPRINT_TRANSFER("try to transfer %u bytes\n", transfer_size);

	do {
#if 0		/* enable this to artificially slow down the transfer.
		 * This makes it much easier to find/reproduce some bugs
		 * occuring occasionally (and in particular during slow transfer)
		 */
		transfer_size = 1000;
#endif

		pipe_tmp = handle->pipe_index;
		bytes_transfered = handle->pipe_index->transfernextstreamblock(
					handle, handle->pipe_index,
					transfer_size);
		handle->pipe_lasttransfer = pipe_tmp;

		if (handle->pipe_index == NULL) {
			gpfStopStream(handle);
			fprintf(stderr, "%s:%s:%i: transfer stopped!\n",
				DEBUG_INFO);
			goto exit;
		}	/* if */

		if (bytes_transfered < 0) {
			/* error occured */
			fprintf(stderr, "unexpected conversion error: %i\n",
				bytes_transfered);
			fprintf(stderr, "transfer stopped\n");
			gpfStopStream(handle);

			handle->pipe_index = handle->pipeline->item_head;

			goto exit;
		}	/* if */

		handle->pipeline->current_transfer += bytes_transfered;

		/* status indicator */
		if (cbI != NULL) {
			cbI(handle->pipeline->current_transfer,
				handle->pipeline->total_transfer);
		}	/* if */

		sum_bytes_transfered += bytes_transfered;

		GPFDPRINT_TRANSFER("%i bytes totally transfered until here\n",
				handle->pipeline->current_transfer);
		transfer_size -= bytes_transfered * bytes_transfered;
	} while (transfer_size > 0);

	GPFDPRINT_TRANSFER("gpfDoNextStreamDataBlock(): leave this function and let the app do something\n");
#if 0
	/* enable this to reproduce the artefacts more easily */
	ggCurTime(&handle->tv);
#endif
exit:
	return sum_bytes_transfered;
}	/* gpfDoNextStreamDataBlock */
