/*
   ******************************************************************************

   LIBGPF: initialization   

   Copyright (C) 2000-2002 Christoph Egger	[Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

   ******************************************************************************
 */


#include <stdio.h>
#include <string.h>
#include <stdarg.h>

#include <ggi/internal/gpf.h>
#include <ggi/internal/gpf_internal.h>
#include <ggi/internal/gpf_debug.h>
#include <ggi/gg.h> /* We use LibGG to manage config files */
#include "config.h"


/* Exported variables */
EXPORTVAR uint32	_gpfDebugState = 0;
EXPORTVAR int		_gpfDebugSync = 0;
EXPORTVAR void		*_gpf_global_lock = NULL;

/* Global variables */
void *_gpfConfigHandle = NULL;

/* Static variables 
 */
static int _gpfLibIsUp		= 0;
static char _gpfconfstub[512]	= GPFCONFDIR;
static char *_gpfconfdir	= _gpfconfstub + GPFTAGLEN;



/* Static variables */
static struct {
	void		*mutex;		/* Lock when changing.. */
	int		handles;	/* # of handles active */
	gpf_handle_t	handle;		/* Linked list of all handles */
} _gpfHandles;	/* This is for remembering handles. */


/*
 * Returns the directory where global config files are kept
 */

const char *gpfGetConfDir(void)
{
#ifdef __WIN32__
	/* On Win32 we allow overriding of the compiled in path. */
	const char *envdir = getenv("GPF_CONFDIR");
	if (envdir) return envdir;
#endif
	return _gpfconfdir;
}	/* gpfGetConfDir */



/* Initialize the library
 */
int gpfInit(void)
{
	int err = 0;
	const char *str, *confdir;
	char *conffile;


	_gpfLibIsUp++;
	if (_gpfLibIsUp > 1) return 0;	/* Initialize only at first call. */

	if ((_gpfHandles.mutex = ggLockCreate()) == NULL) {
		fprintf(stderr, "LibGPF: unable to initialize core mutex.\n");
		return GGI_EUNKNOWN;
	}	/* if */

	if ((_gpf_global_lock = ggLockCreate()) == NULL) {
		fprintf(stderr,"LibGPF: unable to initialize global mutex.\n");
		ggLockDestroy(_gpfHandles.mutex);
		return GGI_EUNKNOWN;
	}	/* if */

	_gpfHandles.handles = 0;
	_gpfHandles.handle = NULL;


	str = getenv("GPF_DEBUG");
	if (str != NULL) {
		_gpfDebugState = atoi(str);
		GPFDPRINT_CORE("Debugging=%d\n", _gpfDebugState);
	}	/* if */

	str = getenv("GPF_DEBUGSYNC");
	if (str != NULL) {
		_gpfDebugSync = 1;
	}	/* if */


	confdir = gpfGetConfDir();
	conffile = malloc(strlen(confdir) + 1 + strlen(GPFCONFFILE)+1);
	if (!conffile) {
		fprintf(stderr, "LibGPF: unable to allocate memory for config filename.\n");
	} else {
#ifdef HAVE_SNPRINTF
		snprintf(conffile, strlen(confdir) + strlen(GPFCONFFILE) + 2,
			"%s/%s", confdir, GPFCONFFILE);
#else
		sprintf(conffile, "%s/%s", confdir, GPFCONFFILE);
#endif
		err = ggLoadConfig(conffile, &_gpfConfigHandle);
		if (err == GGI_OK) {
			free(conffile);
			return GGI_OK;
		}	/* if */

		fprintf(stderr,"LibGPF: couldn't open %s.\n", conffile);
		free(conffile);
	}	/* if */

	ggLockDestroy(_gpfHandles.mutex);
	ggLockDestroy(_gpf_global_lock);
	_gpfLibIsUp--;

	return err;
}	/* gpfInit */


/* Deinitialize the extension
 */
int gpfExit(void)
{
	GPFDPRINT_CORE("gpfExit called\n");
	if (!_gpfLibIsUp) return GGI_ENOTALLOC;

	if (_gpfLibIsUp > 1) {
		/* Exit only at last call */
		_gpfLibIsUp--;
		return _gpfLibIsUp;
	}	/* if */

	GPFDPRINT_CORE("gpfExit: really destroying.\n");
	while (_gpfHandles.handle != NULL) gpfClose(_gpfHandles.handle);

	ggLockDestroy(_gpfHandles.mutex);
	ggLockDestroy(_gpf_global_lock);


	ggFreeConfig(_gpfConfigHandle);
	_gpfLibIsUp = 0;

	GPFDPRINT_CORE("gpfExit: done!\n");
	return 0;

}	/* gpfExit */


void gpfPanic(const char *format,...)
{
	va_list ap;

	GPFDPRINT_CORE("gpfPanic called\n");

	va_start(ap,format);
	vfprintf(stderr,format,ap);
	fflush(stderr);
	va_end(ap);

	while (gpfExit() > 0);	/* kill all instances ! */
	exit(1);
}	/* gpfPanic */


/* Opens a handle.
 */
static int vgpfOpenInput(gpf_handle_t *handle, const char *input,
			const char *argf, va_list ap)
{
	gpf_handle_t hnd;
	int rc = GGI_OK;
	int inited = 0;


	if (!_gpfLibIsUp) return -1;

	GPFDPRINT_CORE("vgpfOpenInput(%p, \"%s\", %s) called\n",
			handle, input, argf);

	if (handle[0] == NULL) {
		if ((handle[0] = _gpfNewHandle()) == NULL) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */

		inited = 1;
	}	/* if */

	hnd = handle[0];

	if (hnd->opio_input != NULL) {
		/* input already opened */
		rc = GGI_EBUSY;
		goto err0;
	}	/* if */


	/* input
	 */

	if (input == NULL) {
		GPFDPRINT_CORE("use auto-target to autodetect "
				"the input-target\n");
		input = "auto";
	}	/* if */

	/* Try the target.. */
	rc = _gpfOpenTarget(hnd, input, argf, ap, GPF_DLTYPE_INPUT);
	if (rc != GGI_OK) {
		GPFDPRINT_CORE("%s:%s:%i: _gpfOpenTarget failed opening "
				" the input target (error code: %i)\n",
				DEBUG_INFO, rc);
		goto err1;
	}	/* if */

	GPFDPRINT_CORE("input-target is %s at address (%p)\n",
			input, hnd->opio_input);

	if (inited && (!hnd->mutex_valid)) {
		GPFDPRINT_CORE("setting up mutexes\n");

		ggLock(_gpfHandles.mutex);
		hnd->next = _gpfHandles.handle;
		_gpfHandles.handle = hnd;
		_gpfHandles.handles++;
		ggUnlock(_gpfHandles.mutex);

		hnd->mutex_valid = 1;
		GPFDPRINT_CORE("setting up mutexes done\n");
	}	/* if */


	GPFDPRINT_CORE("vgpfOpenInput returns successful (handle: %p)\n",
			hnd);

	LIB_ASSERT(rc == GGI_OK);
	return GGI_OK;

err1:
	if (hnd->mutex_valid) {
		ggLock(_gpfHandles.mutex);
		_gpfHandles.handles--;
		_gpfHandles.handle = hnd->next;
		ggUnlock(_gpfHandles.mutex);
		GPFDPRINT_CORE("mutexes destroyed\n");
	}	/* if */

	_gpfDestroyHandle(hnd);
err0:

	GPFDPRINT_CORE("vgpfOpenInput: return with failure (error: %i)\n",
			rc);
	return rc;
}	/* vgpfOpenInput */

int gpfOpenInput(gpf_handle_t *handle, const char *input,
		const char *argf, ...)
{
	int rc;
	va_list ap;

	va_start(ap, argf);
	rc = vgpfOpenInput(handle, input, argf, ap);
	va_end(ap);

	return rc;
}	/* gpfOpenInput */


static int vgpfOpenOutput(gpf_handle_t *handle, const char *output,
			 const char *argf, va_list ap)
{
	gpf_handle_t hnd;
	int rc = GGI_OK;
	int inited = 0;

	if (!_gpfLibIsUp) return -1;

	GPFDPRINT_CORE("vgpfOpenInput(%p, \"%s\", %s) called\n",
			handle, output, argf);

	if (handle[0] == NULL) {
		if ((handle[0] = _gpfNewHandle()) == NULL) {
			rc = GGI_ENOMEM;
			goto err0;
		}	/* if */

		inited = 1;
	}	/* if */

	hnd = handle[0];

	if (hnd->opio_output != NULL) {
		/* output already opened */
		rc = GGI_EBUSY;
		goto err0;
	}	/* if */


	/* output
	 */

	if (output == NULL) {
		GPFDPRINT_CORE("use auto-target to autodetect "
				"the output-target\n");
		output = "auto";
	}	/* if */

	/* Try the target.. */
	rc = _gpfOpenTarget(hnd, output, argf, ap, GPF_DLTYPE_OUTPUT);
	if (rc != GGI_OK) {
		GPFDPRINT_CORE("%s:%s:%i: _gpfOpenTarget failed opening "
				" the output target (error code: %i)\n",
				DEBUG_INFO, rc);
		goto err1;
	}	/* if */

	GPFDPRINT_CORE("output-target is %s at address (%p)\n",
			output, hnd->opio_output);

	if (inited && (!hnd->mutex_valid)) {
		GPFDPRINT_CORE("setting up mutexes\n");

		ggLock(_gpfHandles.mutex);
		hnd->next = _gpfHandles.handle;
		_gpfHandles.handle = hnd;
		_gpfHandles.handles++;
		ggUnlock(_gpfHandles.mutex);

		hnd->mutex_valid = 1;
		GPFDPRINT_CORE("setting up mutexes done\n");
	}	/* if */


	GPFDPRINT_CORE("vgpfOpenOutput returns successful (handle: %p)\n",
			hnd);
	LIB_ASSERT(rc == GGI_OK);
	return GGI_OK;

err1:
	if (hnd->mutex_valid) {
		ggLock(_gpfHandles.mutex);
		_gpfHandles.handles--;
		_gpfHandles.handle = hnd->next;
		ggUnlock(_gpfHandles.mutex);
		GPFDPRINT_CORE("mutexes destroyed\n");
	}	/* if */

	_gpfDestroyHandle(hnd);
err0:
	GPFDPRINT_CORE("vgpfOpenOutput: return with failure (error: %i)\n",
			rc);
	return rc;
}	/* vgpfOpenOutput */


int gpfOpenOutput(gpf_handle_t *handle, const char *output,
		const char *argf, ...)
{
	int rc;
	va_list ap;

	va_start(ap, argf);
	rc = vgpfOpenOutput(handle, output, argf, ap);
	va_end(ap);

	return rc;
}	/* gpfOpenOutput */





gpf_handle_t gpfOpen(const char *input, const char *inputargf,
		const char *output, const char *outputargf, ...)
{
	va_list ap;
	gpf_handle_t hnd = NULL;
	int rc;


	GPFDPRINT_CORE("gpfOpen(%s, %s, %s, %s) called\n",
			input, inputargf, output, outputargf);

	va_start(ap, outputargf);

	rc = vgpfOpenInput(&hnd, input, inputargf, ap);
	if (rc != GGI_OK) {
		goto err0;
	}	/* if */

	_gpfGetLastArg(inputargf, &ap);

	rc = vgpfOpenOutput(&hnd, output, outputargf, ap);
	if (rc != GGI_OK) {
		goto err1;
	}	/* if */
	
	va_end(ap);


	GPFDPRINT_CORE("gpfOpen: returning %p\n", hnd);

	return hnd;

err1:
	rc = _gpfCloseTarget(hnd, GPF_DLTYPE_INPUT);
	LIB_ASSERT(rc == GGI_OK);

err0:
	GPFDPRINT_CORE("gpfOpen: failure\n");
	return NULL;
}	/* gpfOpen */





int gpfCloseInput(gpf_handle_t handle)
{
	int rc;

	if (!gpfIsStopped(handle)) {
		/* Don't close target, while transfering data */
		rc = GGI_EBUSY;
		goto exit;
	}	/* if */

	rc =_gpfCloseTarget(handle, GPF_DLTYPE_INPUT);
        LIB_ASSERT(rc == GGI_OK);

	if (!gpfIsOpened(handle, GPF_DLTYPE_OUTPUT)) {
		rc = gpfClose(handle);
		LIB_ASSERT(rc == GGI_OK);
	}	/* if */

exit:
	return rc;
}	/* gpfCloseInput */


int gpfCloseOutput(gpf_handle_t handle)
{
	int rc;

	if (!gpfIsStopped(handle)) {
		/* Don't close target, while transfering data */
		rc = GGI_EBUSY;
		goto exit;
	}	/* if */

        rc = _gpfCloseTarget(handle, GPF_DLTYPE_OUTPUT);
        LIB_ASSERT(rc == GGI_OK);

	if (!gpfIsOpened(handle, GPF_DLTYPE_INPUT)) {
		rc = gpfClose(handle);
		LIB_ASSERT(rc == GGI_OK);
	}	/* if */

exit:
	return rc;
}	/* gpfCloseOutput */


int gpfIsOpened(gpf_handle_t handle, int type)
{
	LIB_ASSERT(handle != NULL);

	if (type & GPF_DLTYPE_INPUT) {
		return (handle->opio_input != NULL) ? 1 : 0;
	}	/* if */
	if (type & GPF_DLTYPE_OUTPUT) {
		return (handle->opio_output != NULL) ? 1 : 0;
	}	/* if */

	return 0;
}	/* gpfIsOpened */




/* gpfClose
 *	Closes the requested handle
 *	Returns 0 on success, < 0 on error
 */
int gpfClose(gpf_handle_t handle)
{
	gpf_handle_t hnd,phnd = NULL;

	GPFDPRINT_CORE("gpfClose(\"%p\") called\n", handle);
	if (!_gpfLibIsUp) return GGI_ENOTALLOC;

	GPFDPRINT_CORE("gpfClose: closing\n");

	for (hnd = _gpfHandles.handle; hnd != NULL; phnd=hnd, hnd=hnd->next) {
		if (hnd == handle) break;
	}	/* for */

	if (hnd == NULL) return GGI_EARGINVAL;

	GPFDPRINT_CORE("lock mutex (%p)\n", _gpfHandles.mutex);
	ggLock(_gpfHandles.mutex);

	if (phnd == NULL) {
		_gpfHandles.handle = hnd->next;
	} else {
		phnd->next = hnd->next;
	}	/* if */

	_gpfHandles.handles--;

	GPFDPRINT_CORE("unlock mutex (%p)\n", _gpfHandles.mutex);
	ggUnlock(_gpfHandles.mutex);


	GPFDPRINT_CORE("destroy handle (%p)\n", hnd);
	_gpfDestroyHandle(hnd);

	GPFDPRINT_CORE("gpfClose: done!\n");

	return 0;
}	/* gpfClose */







/* This should be included by exactly _one_ file in each library. */
#include <ggi/internal/ggilibinit.h>
