/*
******************************************************************************

   LibGPF API header file

   Copyright (C) 2000      Andreas Beck		[becka@ggi-project.org]
   Copyright (C) 2000-2001 Christoph Egger	[Christoph_Egger@t-online.de]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/


#ifndef _GPF_H
#define _GPF_H


#include <stdio.h>
#include <stdlib.h>



/* Startup/Shutdown:
 */

int gpfInit(void);
int gpfExit(void);
void gpfPanic(const char *format,...);


/* Structure
 */

#ifdef HAVE_STRUCT_GPF
typedef struct gpf_handle *gpf_handle_t;
#else
struct gpf_handle { };

typedef struct gpf_handle *gpf_handle_t;
#endif

#include <ggi/gpf_structs.h>



/* Opens a source and a destination target.
 * The syntax is this:
 *	"<input-target>", "<input-target-options>",
 *	"<output-target>", "<output-target-options>"
 * 
 * <target>: The io-target used for input (source) / output (sink)
 * <target-options>: Here you can tell the io-target, what to do.
 *		Note, that this format is target dependent.
 * 
 * 1. Example how to load an avi-video from an webserver through a proxy
 * into a libbuf object, where a gstreamer avi-plugin is used to do this:
 *
 * ggiBuf_t buf = ggiBufCreateBuffer(vis, ...);
 * ...
 * handle = gpfOpen("gstreamer", "-location=www-proxy(%s) -plugin=avi",
 *		"libbuf", "-buf=%p",
 *		"http://www.avi-collection.org/intro.avi", buf);
 *
 * 2. Example how to load a gif-picture from local disk into a GGI visual:
 *
 * vis = ggiOpen("memory");
 * ...
 * handle = gpfOpen("gif",
 *	"-location=file:/usr/local/ftp/pub/picture-collection/GGI_logo.gif",
 *	"libggi", "-vis=%p", vis);
 *
 * 3. Example how to save a jpg-picture from GGI visual
 * to a remote ftpserver.
 *
 * handle = gpfOpen("libggi","-vis=%p",
 *	"jpg", "-location=ftp://ftp.picture-collection.org/pub/image.jpg",
 *	vis);
 */

gpf_handle_t gpfOpen(const char *input, const char *inputargf,
		     const char *output, const char *outputargf, ...);

int gpfOpenInput(gpf_handle_t *handle, const char *input, const char *argf, ...);
int gpfOpenOutput(gpf_handle_t *handle, const char *output, const char *argf, ...);


/* Sets up the pipeline. It fails, when no pipeline couldn't be set up.
 */
int gpfPipelineSetup(gpf_handle_t handle);


/* Shuts down the pipeline, closes the handle,
 * flush buffers and free ressources.
 */
int gpfClose (gpf_handle_t handle);

int gpfCloseInput(gpf_handle_t handle);
int gpfCloseOutput(gpf_handle_t handle);


#define GPF_DLTYPE_INPUT	0x01
#define GPF_DLTYPE_OUTPUT	0x02

int gpfIsOpened(gpf_handle_t handle, int type);



/* for multiimage Formats (animated gif, interlaced jpeg, video, ...)
 */
int gpfSetCurrFrame(gpf_handle_t handle, int num_frame);
int gpfGetCurrFrae(gpf_handle_t handle, int *num_frame);
int gpfGetNumFrames(gpf_handle_t handle, int *num_frames);


/* transfer rate
 */
int gpfGetTransferRate(gpf_handle_t handle);


/* properties. Changing properties have only an effect, when
 * data-transfer is stopped.
 */
int gpfClearProperties(struct gpf_properties_t *props);

int gpfSetProperties(gpf_handle_t handle, const struct gpf_properties_t *props);
int gpfGetProperties(gpf_handle_t handle, struct gpf_properties_t *props);


#if 0
/* This function isn't implemented well yet and doesn't work therefore.
 * This function was intended used by libvideo to determine the sound format
 * of a video. But because libvideo's design isn't done, it is of no use yet.
 * Until libvideo's design is done, this function is disabled and should
 * never be used.
 */

/* get the most important file data
 */
int gpfGetFileInfo(gpf_handle_t handle, struct gpf_fileinfo_t **info);
#endif


typedef void (*gpfIndicatorCallback)(size_t current, size_t total);


int gpfStartStream(gpf_handle_t handle);
int gpfStopStream(gpf_handle_t handle);

ssize_t gpfTransferNextStreamBlock(gpf_handle_t handle, gpfIndicatorCallback cbI);

int gpfIsStopped(gpf_handle_t handle);
int gpfIsEOS(gpf_handle_t handle);



/* Helper functions to access the property structure.
 */

#define gpfAddFlags(props, flags)	\
	gpfSetFlags(props, gpfGetFlags(props) | flags);

#define gpfRemoveFlags(props, flags)	\
	gpfSetFlags(props, gpfGetFlags(props) & ~flags);

enum gpf_flags gpfSetFlags(struct gpf_properties_t *props,
			enum gpf_flags flags);
enum gpf_flags gpfGetFlags(const struct gpf_properties_t *props);


int gpfSetTransferRate(struct gpf_properties_t *props, int rate);



#endif	/* _GPF_H */
