/*
 ******************************************************************************

 LIBGPF: ppm io target pre/post conversion functions

 Copyright (C) 2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "ppm_gpf.h"


ssize_t GPF_PPM_transfernextstreamblock_input(gpf_handle_t handle,
					struct gpf_pipeline_item_t *pipe,
					size_t transfer_size)
{
	ssize_t rc;
	struct gpf_pipeline_t *pipeline = NULL;
	struct gpf_handle_opio_t *opio = NULL;
	struct ppmgpf_priv *priv = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(pipe != NULL);

	GPFDPRINT_TRANSFER("io-ppm (input): used in handle %p, pipe: %p\n",
			handle, pipe);

	pipeline = handle->pipeline;
	opio = handle->opio_input;
	priv = PPM_PRIV(opio);

	LIB_ASSERT(PIPE_READBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_READBUFSIZE(pipe) > 0);
	LIB_ASSERT(PIPE_WRITEBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_WRITEBUFSIZE(pipe) > 0);

	LIMIT_TRANSFERSIZE;

	LIB_ASSERT(pipe->buf_conv == NULL);
	LIB_ASSERT(pipe->buf_conv_size == 0);
	pipe->buf_conv = PIPE_READBUF(pipe);
	pipe->buf_conv_size = PIPE_READBUFSIZE(pipe);

	/* opio->read performs conversion if needed */
	GPFDPRINT_TRANSFER("io-ppm (input): try to read %i bytes, %i bytes are cached\n",
			transfer_size - PIPE_READCACHESIZE(pipe), PIPE_READCACHESIZE(pipe));
	rc = opio->read(opio, (PIPE_READBUF(pipe) + PIPE_READCACHESIZE(pipe)),
			transfer_size - PIPE_READCACHESIZE(pipe));
	if ((rc == 0) && (PIPE_READCACHESIZE(pipe) == 0)) {
		GPFDPRINT_TRANSFER("io-ppm (input): There's no more data to read\n");
		handle->pipe_index = NULL;
		goto exit;
	}	/* if */
	if (rc < 0) {
		GPFDPRINT("io-ppm (input): error %i occured during reading\n", rc);
		goto exit;
	}	/* if */

	/* Check here, if there's another frame */
	if (priv->is_frame_finished(priv, pipe)) {
		priv->max_frames++;
		GPFDPRINT_TRANSFER("io-ppm (input): there's another frame (%i)\n",
				priv->max_frames);
	}	/* if */

	pipe->readbytes = rc;
	PIPE_WRITEBUFPOS(pipe) = PIPE_WRITECACHESIZE(pipe);

	GPFDPRINT_TRANSFER("io-ppm (input): %i bytes read, %i bytes are cached, try %i bytes to write\n",
			pipe->readbytes, PIPE_READCACHESIZE(pipe),
			pipe->readbytes + PIPE_READCACHESIZE(pipe));
	rc = pipeline->write(pipe,
			pipe->readbytes + PIPE_READCACHESIZE(pipe),
			PIPE_WRITEBUFSIZE(pipe));
	if (rc < 0) {
		GPFDPRINT("io-ppm (input): error %i occured during write\n", rc);
		goto exit;
	}	/* if */

	pipe->writebytes = rc;
	GPFDPRINT_TRANSFER("io-ppm (input): %i bytes finally written, %i bytes to cache\n",
			pipe->writebytes, pipe->readbytes - pipe->writebytes);

	if (pipe->readbytes > pipe->writebytes) {
		PIPE_READCACHEDATA(pipe, pipe->readbytes - pipe->writebytes);
	}	/* if */
	PIPE_READBUFPOS(pipe) = 0;

	handle->pipe_index = handle->pipe_index->next;

exit:
	pipe->buf_conv = NULL;
	pipe->buf_conv_size = 0;

	return rc;
}	/* GPF_PPM_transfernextstreamblock_input */



ssize_t GPF_PPM_transfernextstreamblock_output(gpf_handle_t handle,
					struct gpf_pipeline_item_t *pipe,
					size_t transfer_size)
{
	ssize_t rc;
	size_t tmp_size;
	struct gpf_handle_opio_t *opio = NULL;
	struct gpf_pipeline_t *pipeline = NULL;
	struct ppmgpf_priv *priv = NULL;

	LIB_ASSERT(handle != NULL);
	LIB_ASSERT(pipe != NULL);

	GPFDPRINT_TRANSFER("io-ppm (output): uses handle: %p, pipe: %p\n",
			handle, pipe);

	pipeline = handle->pipeline;
	opio = handle->opio_output;
	priv = PPM_PRIV(opio);

	LIB_ASSERT(PIPE_READBUF(pipe) != NULL);
	LIB_ASSERT(PIPE_READBUFSIZE(pipe) > 0);
	LIB_ASSERT(pipe->prev != NULL);

	opio->pipe->width = pipe->prev->width;
	opio->pipe->height = pipe->prev->height;

	GET_TARGET_TRANSFERSIZE;
	LIMIT_TRANSFERSIZE;
	REALLOC_BUFCONVSIZE(transfer_size);

	GPFDPRINT_TRANSFER("io-ppm (output): try to transfer %i bytes, %i bytes are cached\n",
			transfer_size, PIPE_READCACHESIZE(pipe));
	GPFDPRINT_TRANSFER("io-ppm (output): %i bytes in read buffer (minus cached bytes)\n",
			PIPE_WRITEBUFPOS(pipe->prev));

	tmp_size = transfer_size;
	PIPE_READBUFPOS(pipe) = 0;
	while (tmp_size > 0) {
		rc = pipeline->read(pipe, tmp_size, transfer_size);
		if (rc < 0) {
			fprintf(stderr, "io-ppm (output): error %i during reading occured\n",
				rc);
			return rc;
		}	/* if */
		if (rc == 0) break;	/* ready? */

		pipe->readbytes += rc;
		GPFDPRINT_TRANSFER("io-ppm (output): %i bytes were read, now try to write them\n", rc);

		/* Check here, if there's another frame */
		if (priv->is_frame_finished(priv, pipe)) {
			priv->max_frames++;
			GPFDPRINT_TRANSFER("io-ppm (input): there's another frame (%i)\n",
					priv->max_frames);
		}	/* if */

		/* performs conversion if needed */
		rc = opio->write(opio, pipe->buf_conv, rc);
		if (rc < 0) {
			fprintf(stderr, "io-ppm (output): error %i during writing occured\n",
				rc);
			return rc;
		}	/* if */

		GPFDPRINT_TRANSFER("io-ppm (output): %i bytes written\n", rc);

		pipe->writebytes += rc;
		tmp_size -= rc;
	}	/* while */

	rc = pipe->writebytes;

	GPFDPRINT_TRANSFER("io-ppm (output): %i bytes transfered, %i bytes must be cached\n",
			pipe->writebytes, PIPE_WRITEBUFPOS(pipe->prev) - pipe->writebytes);

	if (PIPE_WRITEBUFPOS(pipe->prev) < pipe->writebytes) {
		PIPE_READCACHEDATA(pipe, pipe->writebytes -
				PIPE_WRITEBUFPOS(pipe->prev) +
				PIPE_READCACHESIZE(pipe));
	} else {
		PIPE_READCACHEDATA(pipe,
				PIPE_WRITEBUFPOS(pipe->prev) -
				pipe->writebytes);
	}	/* if */

	if (PIPE_WRITEBUFPOS(pipe->prev) <= pipe->writebytes) {
		handle->pipe_index = handle->pipe_index->prev;
		pipe->readbytes = pipe->writebytes = 0;
	}	/* if */

	GPFDPRINT_TRANSFER("io-ppm (output): %i bytes cached\n",
			PIPE_READCACHESIZE(pipe));

	return rc;
}	/* GPF_PPM_transfernextstreamblock_output */






ssize_t GPF_PPM_conv_8to16(struct ppmgpf_priv *priv,
			void *buf_in, void *buf_out,
			size_t size_in, size_t size_out)
{
	ssize_t rc = 0;
	size_t conv_bytes;
	unsigned char *bufc_in;
	unsigned short *bufc_out;

	if (size_out < 2) return 0;

	bufc_in = (unsigned char *)buf_in;
	bufc_out = (unsigned short *)buf_out;

	conv_bytes = size_in;
	while (conv_bytes--) {
		bufc_out[0] = (unsigned short)(bufc_in[0] << 8 | bufc_in[0]);
		bufc_in++;
		bufc_out++;
		size_out -= 2;

		if ((size_out == 0) && (conv_bytes > 0)) {
			rc = size_in - conv_bytes;

			GPFDPRINT_TARGET("%s:%s:%i: output buffer not large"
					" enough to complete conversion (%u,%i)\n",
					DEBUG_INFO, size_in, rc);
			return rc;
		}	/* if */
	}	/* while */

	rc = size_in - conv_bytes;
	return rc;
}	/* GPF_PPM_conv_8to16 */



ssize_t GPF_PPM_conv_16to8(struct ppmgpf_priv *priv,
			void *buf_in, void *buf_out,
			size_t size_in, size_t size_out)
{
	ssize_t rc = 0;
	size_t conv_bytes;
	unsigned short *bufc_in;
	unsigned char *bufc_out;

	if (size_out < 1) return 0;

	bufc_in = (unsigned short *)buf_in;
	bufc_out = (unsigned char *)buf_out;

	conv_bytes = size_in;
	while (conv_bytes > 0) {
		bufc_out[0] = (unsigned char)(bufc_in[0] >> 8);
		bufc_in++;
		bufc_out++;
		size_out--;
		conv_bytes -= 2;

		if ((size_out == 0) && (conv_bytes > 0)) {
			rc = size_in - conv_bytes;

			GPFDPRINT_TARGET("%s:%s:%i: output buffer not large"
					" enough to complete conversion (%u,%i)\n",
					DEBUG_INFO, size_in, rc);
			return rc;
		}	/* if */
	}	/* while */

	rc = size_in - conv_bytes;
	return rc;
}	/* GPF_PPM_conv_16to8 */
