/*
 ******************************************************************************

 LIBGPF: rgba io pipe functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "rgba_gpf.h"


#define DOSHIFT(val, shift)	\
	(((shift) >= 0) ? (val) << (shift) : (val) >> -(shift))


static ssize_t _gpfRGBA_conv_rgba2rgba(struct gpf_pipeline_t *pipeline,
				struct gpf_pipeline_item_t *pipe,
				size_t buf_size)
{
	struct rgbagpf_priv *priv;
	size_t conv_bytes;
	size_t readbytes;
	size_t writebytes;
	ssize_t rc;

	priv = RGBA_PRIV(pipe);

	LIB_ASSERT(pipe->buf_conv_size == sizeof(ggi_color));

	pipe->readbytes = pipe->writebytes = 0;
	conv_bytes = buf_size;
	while (conv_bytes > 0) {
		readbytes = pipeline->read(pipe,
					ByPP(priv->pf_input.size),
					buf_size);
		if (readbytes == 0) break;	/* ready ? */

		if (readbytes < ByPP(priv->pf_input.size)) {
			GPFDPRINT_TRANSFER("pipe-rgba: (after reading) read position at %i, "
					"write position at %i, cache %i byte(s)\n",
				PIPE_READBUFPOS(pipe), PIPE_WRITEBUFPOS(pipe), conv_bytes);
			PIPE_READCACHEDATA(pipe, conv_bytes);
			goto exit;
		}	/* if */
		LIB_ASSERT(readbytes == ByPP(priv->pf_input.size));

		pipe->readbytes += readbytes;

		priv->dounmap(priv, pipe->buf_conv,
				ByPP(priv->pf_input.size),
				pipe->buf_conv_size);

		priv->domap(priv, pipe->buf_conv,
				pipe->buf_conv_size,
				ByPP(priv->pf_output.size));

		writebytes = pipeline->write(pipe,
				ByPP(priv->pf_output.size),
				PIPE_WRITEBUFSIZE(pipe));
		if (writebytes < ByPP(priv->pf_output.size)) {
			GPFDPRINT_TRANSFER("pipe-rgba: (after reading) read position at %i, "
					"write position at %i, cache %i byte(s)\n",
				PIPE_READBUFPOS(pipe), PIPE_WRITEBUFPOS(pipe), conv_bytes);
			PIPE_READCACHEDATA(pipe, conv_bytes);
			goto exit;
		}	/* if */

		LIB_ASSERT(writebytes == ByPP(priv->pf_output.size));

		pipe->writebytes += readbytes;
		conv_bytes -= readbytes;
	}	/* while */

	PIPE_READCACHESIZE(pipe) = 0;
	GPFDPRINT_TRANSFER("pipe-rgba: (after reading) read position at %i, "
			"write position at %i, 0 bytes cached\n",
			PIPE_READBUFPOS(pipe), PIPE_WRITEBUFPOS(pipe));

exit:
	GPFDPRINT_TRANSFER("pipe-rgba: %i bytes transfered\n",
			pipe->writebytes);

	rc = pipe->writebytes;
	return rc;
}	/* _gpfRGBA_conv_rgba2rgba */

/* ---------------------------------------------------------------------- */


int GPF_color_TRUE_mapcolor_gte16(struct rgbagpf_priv *rgba, char *buf_map,
				size_t input_size, size_t output_size)
{
	LIB_ASSERT(input_size == sizeof(ggi_color));
	LIB_ASSERT(output_size == sizeof(ggi_color));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	/* nothing to do here as the values are already in the right
	 * format
	 */
	return GGI_OK;
}	/* GPF_color_TRUE_mapcolor_gte16 */


int GPF_color_TRUE_mapcolor(struct rgbagpf_priv *rgba, char *buf_map,
				size_t input_size, size_t output_size)
{
	ggi_color col = { 0,0,0,0};
	ggi_pixel pix = 0;
	color_truepriv *priv = COLOR_TRUEPRIV(rgba);


	LIB_ASSERT(input_size == sizeof(ggi_color));
	LIB_ASSERT(output_size <= sizeof(pix));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	memcpy(&col, buf_map, input_size);

	pix =  (DOSHIFT(col.r, priv->red_map_output) &
		priv->red_mask_output) |
		(DOSHIFT(col.g, priv->green_map_output) &
		priv->green_mask_output) |
		(DOSHIFT(col.b, priv->blue_map_output) &
		priv->blue_mask_output);

	memcpy(buf_map, &pix, output_size);

	GPFDPRINT_CONV("TRUE_mapcolor(%p, {r=0x%X, g=0x%X, b=0x%X}) returning {pixel = 0x%X}\n",
			rgba, col.r, col.g, col.b, pix);

	return GGI_OK;
}	/* GPF_true_mapcolor */


int GPF_color_GREY_mapcolor(struct rgbagpf_priv *rgba, char *buf_map,
			size_t input_size, size_t output_size)
{
	const ggi_color *col;
	ggi_pixel pix = 0;


	LIB_ASSERT(input_size == sizeof(ggi_color));
	LIB_ASSERT(output_size <= sizeof(pix));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	col = (ggi_color *)buf_map;

	/* NB: This formula is not very good... */
	pix = (col->r*82 + col->g*124 + col->b*50)
		>> COLOR_GREYPRIV(rgba)->shift_output;

	memcpy(buf_map, &pix, output_size);

	GPFDPRINT_CONV("GREY_mapcolor(%p, {r=0x%X, g=0x%X, b=0x%X}) returning {pixel = 0x%X}\n",
			rgba, col->r, col->g, col->b, pix);

	return GGI_OK;
}	/* GPF_color_GREY_mapcolor */

/* ---------------------------------------------------------------------- */

int GPF_color_TRUE_unmappixel_gte16(struct rgbagpf_priv *rgba, char *buf_unmap,
			size_t input_size, size_t output_size)
{
	LIB_ASSERT(input_size == sizeof(ggi_color));
	LIB_ASSERT(output_size == sizeof(ggi_color));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	/* nothing to do here as the values are already in the right
	 * format
	 */
	return GGI_OK;
}	/* GPF_color_TRUE_unmappixel_gte16 */


int GPF_color_TRUE_unmappixel_gte8(struct rgbagpf_priv *rgba, char *buf_unmap,
			size_t input_size, size_t output_size)
{
	ggi_color col;
	ggi_pixel pixel = 0;
	color_truepriv *priv = COLOR_TRUEPRIV(rgba);


	LIB_ASSERT(input_size <= sizeof(ggi_pixel));
	LIB_ASSERT(output_size == sizeof(ggi_color));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	memcpy(&pixel, buf_unmap, input_size);

	col.r = DOSHIFT(pixel & priv->red_mask_input,
			priv->red_unmap_input);
	col.r |= col.r >> priv->red_nbits_input;
	col.g = DOSHIFT(pixel & priv->green_mask_input,
			priv->green_unmap_input);
	col.g |= col.g >> priv->green_nbits_input;
	col.b = DOSHIFT(pixel & priv->blue_mask_input,
			priv->blue_unmap_input);
	col.b |= col.b >> priv->blue_nbits_input;


	memcpy(buf_unmap, &col, output_size);

	GPFDPRINT_CONV("TRUE_unmappixel_gte8 (%p, {pixel = 0x%X}) returning {r=0x%X, g=0x%X, b=0x%X}\n",
			rgba, pixel, col.r, col.g, col.b );

	return GGI_OK;
}	/* GPF_color_TRUE_unmappixel_gte8 */


int GPF_color_TRUE_unmappixel_gte4(struct rgbagpf_priv *rgba, char *buf_unmap,
			size_t input_size, size_t output_size)
{
	ggi_color col;
	ggi_pixel pixel = 0;
	color_truepriv *priv = COLOR_TRUEPRIV(rgba);


	LIB_ASSERT(input_size <= sizeof(ggi_pixel));
	LIB_ASSERT(output_size == sizeof(ggi_color));

	memcpy(&pixel, buf_unmap, input_size);

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i, pixel: 0x%X\n",
			DEBUG_INFO, input_size, output_size, pixel);


	col.r = DOSHIFT(pixel & priv->red_mask_input,
			priv->red_unmap_input);
	col.r |= col.r >> priv->red_nbits_input;
	col.r |= col.r >> (priv->red_nbits_input << 1);

	col.g = DOSHIFT(pixel & priv->green_mask_input,
			priv->green_unmap_input);
	col.g |= col.g >> priv->green_nbits_input;
	col.g |= col.g >> (priv->green_nbits_input << 1);

	col.b = DOSHIFT(pixel & priv->blue_mask_input,
			priv->blue_unmap_input);
	col.b |= col.b >> priv->blue_nbits_input;
	col.b |= col.b >> (priv->blue_nbits_input << 1);

	memcpy(buf_unmap, &col, output_size);

	GPFDPRINT_CONV("TRUE_unmappixel_gte4 (%p, {pixel = 0x%X}) returning {r=0x%X, g=0x%X, b=0x%X}\n",
			rgba, pixel, col.r, col.g, col.b );

	return GGI_OK;
}	/* GPF_color_TRUE_unmappixel_gte4 */


int GPF_color_TRUE_unmappixel_gte2(struct rgbagpf_priv *rgba, char *buf_unmap,
			size_t input_size, size_t output_size)
{
	ggi_color col;
	ggi_pixel pixel = 0;
	color_truepriv *priv = COLOR_TRUEPRIV(rgba);


	LIB_ASSERT(input_size <= sizeof(ggi_pixel));
	LIB_ASSERT(output_size == sizeof(ggi_color));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	memcpy(&pixel, buf_unmap, input_size);

	col.r = DOSHIFT(pixel & priv->red_mask_input,
			priv->red_unmap_input);
	col.r |= col.r >> priv->red_nbits_input;
	col.r |= col.r >> (priv->red_nbits_input << 1);
	col.r |= col.r >> (priv->red_nbits_input << 2);
	col.g = DOSHIFT(pixel & priv->green_mask_input,
			priv->green_unmap_input);
	col.g |= col.g >> priv->green_nbits_input;
	col.g |= col.g >> (priv->green_nbits_input << 1);
	col.g |= col.g >> (priv->green_nbits_input << 2);
	col.b = DOSHIFT(pixel & priv->blue_mask_input,
			priv->blue_unmap_input);
	col.b |= col.b >> priv->blue_nbits_input;
	col.b |= col.b >> (priv->blue_nbits_input << 1);
	col.b |= col.b >> (priv->blue_nbits_input << 2);


	memcpy(buf_unmap, &col, output_size);

	GPFDPRINT_CONV("TRUE_unmappixel_gte2 (%p, {pixel = 0x%X}) returning {r=0x%X, g=0x%X, b=0x%X}\n",
			rgba, pixel, col.r, col.g, col.b );

	return GGI_OK;
}	/* GPF_color_TRUE_unmappixel_gte2 */


/* For the rare but extremely painful cases. */
int GPF_color_TRUE_unmappixel_gte1(struct rgbagpf_priv *rgba, char *buf_unmap,
			size_t input_size, size_t output_size)
{
	ggi_color col;
	ggi_pixel pixel = 0;
	color_truepriv *priv = COLOR_TRUEPRIV(rgba);


	LIB_ASSERT(input_size <= sizeof(ggi_pixel));
	LIB_ASSERT(output_size == sizeof(ggi_color));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	memcpy(&pixel, buf_unmap, input_size);

	if (priv->red_nbits_input != 1) {
		col.r = DOSHIFT(pixel & priv->red_mask_input,
				priv->red_unmap_input);
		col.r |= col.r >> priv->red_nbits_input;
		col.r |= col.r >> (priv->red_nbits_input << 1);
		col.r |= col.r >> (priv->red_nbits_input << 2);
	} else col.r = (pixel & priv->red_mask_input) ? 0xffff : 0x0000;

	if (priv->green_nbits_input != 1) {
		col.g = DOSHIFT(pixel & priv->green_mask_input,
				priv->green_unmap_input);
		col.g |= col.g >> priv->green_nbits_input;
		col.g |= col.g >> (priv->green_nbits_input << 1);
		col.g |= col.g >> (priv->green_nbits_input << 2);
	} else col.g = (pixel & priv->green_mask_input) ? 0xffff : 0x0000;

	if (priv->blue_nbits_input != 1) {
		col.b = DOSHIFT(pixel & priv->blue_mask_input,
				priv->blue_unmap_input);
		col.b |= col.b >> priv->blue_nbits_input;
		col.b |= col.b >> (priv->blue_nbits_input << 1);
		col.b |= col.b >> (priv->blue_nbits_input << 2);
	} else col.b = (pixel & priv->blue_mask_input) ? 0xffff : 0x0000;


	memcpy(buf_unmap, &col, output_size);

	GPFDPRINT_CONV("TRUE_unmappixel_gte1 (%p, {pixel = 0x%X}) returning {r=0x%X, g=0x%X, b=0x%X}\n",
			rgba, pixel, col.r, col.g, col.b );

	return GGI_OK;
}	/* GPF_color_TRUE_unmapping_gte1 */


int GPF_color_GREY_unmappixel(struct rgbagpf_priv *rgba, char *buf_unmap,
			size_t input_size, size_t output_size)
{
	ggi_color col;
	ggi_pixel pixel = 0;


	LIB_ASSERT(input_size <= sizeof(ggi_pixel));
	LIB_ASSERT(output_size == sizeof(ggi_color));

	GPFDPRINT_CONV("%s:%s:%i: input_size: %i, output_size: %i\n",
			DEBUG_INFO, input_size, output_size);

	memcpy(&pixel, buf_unmap, input_size);

	col.r = col.g = col.b =
		(pixel << COLOR_GREYPRIV(rgba)->shift_input) >> 8;

	memcpy(buf_unmap, &col, output_size);

	GPFDPRINT_CONV("GREY_unmappixel (%p, {pixel = 0x%X}) returning {r=0x%X, g=0x%X, b=0x%X}\n",
			rgba, pixel, col.r, col.g, col.b );

	return GGI_OK;
}	/* GPF_color_GREY_unmappixel */






static int _gpf_mask2shift(uint32 mask)
{
	int shift;
	for (shift = 32; mask != 0; mask >>= 1) {
		shift--;
	}	/* for */
	if (shift == 32) shift = 0;

	return shift;
}	/* _gpf_mask2shift */


static int calc_total(ggi_pixel mask)
{
	int total;

	for (total=0; mask != 0; mask >>= 1, total++) {
	}	/* for */

	return total;
}	/* calc_total */


static int calc_nbits(ggi_pixel mask)
{
	int nbits;

	while (!(mask & 0x0001)) mask >>= 1;

	for (nbits=0; mask != 0; mask >>= 1, nbits++) {
	}	/* for */

	return nbits;
}	/* calc_nbits */




int GPF_RGBA_setup(gpf_handle_t handle,
		struct gpf_pipeline_item_t *pipe)
{
	struct rgbagpf_priv *priv;
	color_greypriv *col_greypriv;
	color_truepriv *col_truepriv;

	int redtot_input = 0;
	int greentot_input = 0;
	int bluetot_input = 0;

	int redtot_output = 0;
	int greentot_output = 0;
	int bluetot_output = 0;


	priv = RGBA_PRIV(pipe);
	col_greypriv = COLOR_GREYPRIV(priv);
	col_truepriv = COLOR_TRUEPRIV(priv);


	GPFDPRINT("pipe-rgba: setup conversion (0x%X -> 0x%X)\n",
		priv->pf_input.flags, priv->pf_output.flags);

	GPFDPRINT_CONV("  input mask (r: %X, g: %X, b: %X)\n",
			priv->pf_input.type.rgba.red_mask,
			priv->pf_input.type.rgba.green_mask,
			priv->pf_input.type.rgba.blue_mask);
	GPFDPRINT_CONV("  output mask (r: %X, g: %X, b: %X)\n",
			priv->pf_output.type.rgba.red_mask,
			priv->pf_output.type.rgba.green_mask,
			priv->pf_output.type.rgba.blue_mask);



	/* Color mapping - input
	 */
	if (priv->pf_input.flags & GPF_PIPE_DF_GREYSCALE) {
		col_greypriv->shift_input = 24 - priv->pf_input.depth;
		priv->dounmap = GPF_color_GREY_unmappixel;

		GPFDPRINT_CONV("  input: shift_input: %i\n",
			col_greypriv->shift_input);

	} else {
		redtot_input   = calc_total(priv->pf_input.type.rgba.red_mask);
		greentot_input = calc_total(priv->pf_input.type.rgba.green_mask);
		bluetot_input  = calc_total(priv->pf_input.type.rgba.blue_mask);

		col_truepriv->red_unmap_input	= 16 - redtot_input;
		col_truepriv->red_mask_input	= priv->pf_input.type.rgba.red_mask;
		col_truepriv->red_nbits_input	= calc_nbits(priv->pf_input.type.rgba.red_mask);

		col_truepriv->green_unmap_input	= 16 - greentot_input;
		col_truepriv->green_mask_input	= priv->pf_input.type.rgba.green_mask;
		col_truepriv->green_nbits_input	= calc_nbits(priv->pf_input.type.rgba.green_mask);

		col_truepriv->blue_unmap_input	= 16 - bluetot_input;
		col_truepriv->blue_mask_input	= priv->pf_input.type.rgba.blue_mask;
		col_truepriv->blue_nbits_input	= calc_nbits(priv->pf_input.type.rgba.blue_mask);

		GPFDPRINT_CONV("  input: tot (r: 0x%X, g: 0x%X, b: 0x%X), "
			"unmap (r: 0x%X, g: 0x%X, b: 0x%X), "
			"mask (r: 0x%X, g: 0x%X, b: 0x%X), "
			"nbits (r: 0x%X, g: 0x%X, b: 0x%X)\n",
			redtot_input, greentot_input, bluetot_input,

			col_truepriv->red_unmap_input,
			col_truepriv->green_unmap_input,
			col_truepriv->blue_unmap_input,

			col_truepriv->red_mask_input,
			col_truepriv->green_mask_input,
			col_truepriv->blue_mask_input,

			col_truepriv->red_nbits_input,
			col_truepriv->green_nbits_input,
			col_truepriv->blue_nbits_input );

		if (COLOR_TRUEPRIV(priv)->red_nbits_input >= 16 &&
		    COLOR_TRUEPRIV(priv)->green_nbits_input >= 16 &&
		    COLOR_TRUEPRIV(priv)->blue_nbits_input >= 16)
		{
			priv->dounmap = GPF_color_TRUE_unmappixel_gte16;
		} else if (COLOR_TRUEPRIV(priv)->red_nbits_input >= 8 &&
		    COLOR_TRUEPRIV(priv)->green_nbits_input >= 8 &&
		    COLOR_TRUEPRIV(priv)->blue_nbits_input >= 8)
		{
			priv->dounmap = GPF_color_TRUE_unmappixel_gte8;
		} else if (COLOR_TRUEPRIV(priv)->red_nbits_input >= 4 &&
		    COLOR_TRUEPRIV(priv)->green_nbits_input >= 4 &&
		    COLOR_TRUEPRIV(priv)->blue_nbits_input >= 4)
		{
			priv->dounmap = GPF_color_TRUE_unmappixel_gte4;
		} else if (COLOR_TRUEPRIV(priv)->red_nbits_input >= 2 &&
		    COLOR_TRUEPRIV(priv)->green_nbits_input >= 2 &&
		    COLOR_TRUEPRIV(priv)->blue_nbits_input >= 2)
		{
			priv->dounmap = GPF_color_TRUE_unmappixel_gte2;
		} else {
			priv->dounmap = GPF_color_TRUE_unmappixel_gte1;
		}	/* if */
	}	/* if */



	/* Color mapping - output
	 */
	if (priv->pf_output.flags & GPF_PIPE_DF_GREYSCALE) {
		col_greypriv->shift_output = 24 - priv->pf_output.depth;
		priv->domap = GPF_color_GREY_mapcolor;

		GPFDPRINT_CONV("  output: shift_output: %i\n",
			col_greypriv->shift_output);
	} else {
		redtot_output   = calc_total(priv->pf_output.type.rgba.red_mask);
		greentot_output = calc_total(priv->pf_output.type.rgba.green_mask);
		bluetot_output  = calc_total(priv->pf_output.type.rgba.blue_mask);

		col_truepriv->red_map_output	= redtot_output - 16;
		col_truepriv->red_mask_output	= priv->pf_output.type.rgba.red_mask;
		col_truepriv->red_nbits_output	= calc_nbits(priv->pf_output.type.rgba.red_mask);

		col_truepriv->green_map_output	= greentot_output - 16;
		col_truepriv->green_mask_output	= priv->pf_output.type.rgba.green_mask;
		col_truepriv->green_nbits_output= calc_nbits(priv->pf_output.type.rgba.green_mask);

		col_truepriv->blue_map_output	= bluetot_output - 16;
		col_truepriv->blue_mask_output	= priv->pf_output.type.rgba.blue_mask;
		col_truepriv->blue_nbits_output	= calc_nbits(priv->pf_output.type.rgba.blue_mask);

		GPFDPRINT_CONV("  output: tot (r: 0x%X, g: 0x%X, b: 0x%X), "
			"map (r: 0x%X, g: 0x%X, b: 0x%X), "
			"mask (r: 0x%X, g: 0x%X, b: 0x%X)\n",
			redtot_output, greentot_output, bluetot_output,

			col_truepriv->red_map_output,
			col_truepriv->green_map_output,
			col_truepriv->blue_map_output,

			col_truepriv->red_mask_output,
			col_truepriv->green_mask_output,
			col_truepriv->blue_mask_output);

		if (COLOR_TRUEPRIV(priv)->red_nbits_output >= 16 &&
		    COLOR_TRUEPRIV(priv)->green_nbits_output >= 16 &&
		    COLOR_TRUEPRIV(priv)->blue_nbits_output >= 16)
		{
			priv->domap = GPF_color_TRUE_mapcolor_gte16;
		} else {
			priv->domap = GPF_color_TRUE_mapcolor;
		}	/* if */
	}	/* if */

	priv->doconversion = _gpfRGBA_conv_rgba2rgba;

	return GGI_OK;
}	/* GPF_RGBA_setup */
