/*
 ******************************************************************************

 LIBGPF: rgba io pipe functions

 Copyright (C) 2001-2002 Christoph Egger	[Christoph_Egger@t-online.de]

 Permission is hereby granted, free of charge, to any person obtaining a
 copy of this software and associated documentation files (the "Software"),
 to deal in the Software without restriction, including without limitation
 the rights to use, copy, modify, merge, publish, distribute, sublicense,
 and/or sell copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following conditions:

 The above copyright notice and this permission notice shall be included in
 all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

 ******************************************************************************
 */

#include "rgba_gpf.h"


ssize_t GPF_RGBA_transfernextstreamblock(gpf_handle_t handle,
				struct gpf_pipeline_item_t *pipe,
				size_t transfer_size)
{
	struct gpf_pipeline_t *pipeline;
	struct rgbagpf_priv *priv;
	ssize_t rc;
	size_t write_size;


	pipeline = handle->pipeline;

	GPFDPRINT_TRANSFER("pipe-rgba: transfer next chunk\n");
	priv = RGBA_PRIV(pipe);


	LIB_ASSERT(pipe->prev != NULL);

	pipe->width = pipe->prev->width;
	pipe->height = pipe->prev->height;

	/* the amount of memory, that we have to transfer */
	LIB_ASSERT(pipe->prev != NULL);

	GET_PIPE_TRANSFERSIZE;
	LIMIT_TRANSFERSIZE;

	/* The size to make the conversion possible */
	REALLOC_BUFCONVSIZE(sizeof(ggi_color));

	LIB_ASSERT(PIPE_READBUFSIZE(pipe) >= priv->input_data_size);

	if ((ByPP(priv->pf_input.size) == ByPP(priv->pf_output.size))
	  || (ByPP(priv->pf_input.size) == sizeof(ggi_color)))
	{
		write_size = transfer_size;
	} else {
		write_size = transfer_size * ByPP(priv->pf_output.size)
				/ ByPP(priv->pf_input.size);
	}	/* if */


	rc = _gpf_realloc_writebuf(pipe, write_size);
	if (rc != GGI_OK) {
		goto err0;
	}	/* if */

	GPFDPRINT_TRANSFER("pipe-rgba: input/output buf-size (%i, %i), "
			"priv->io_data_size (%i, %i), transfer_size (%i)\n",
			PIPE_READBUFSIZE(pipe),PIPE_WRITEBUFSIZE(pipe),
			priv->input_data_size, priv->output_data_size,
			transfer_size);

	PIPE_WRITEBUFPOS(pipe) = PIPE_WRITECACHESIZE(pipe);
	PIPE_READBUFPOS(pipe) = 0;

	GPFDPRINT_TRANSFER("pipe-rgba: %i bytes cached in the next pipe, "
			" %i bytes cached in pipe-rgba, "
			"set write position to %i\n",
			PIPE_WRITECACHESIZE(pipe), PIPE_READCACHESIZE(pipe),
			PIPE_WRITEBUFPOS(pipe));

	rc = priv->doconversion(pipeline, pipe,	transfer_size);

	if (rc < 0) {
		GPFDPRINT("%s:%s:%i: error after conversion - rc: %i\n",
				DEBUG_INFO, rc);
		goto err1;
	}	/* if */

	LIB_ASSERT(pipe->writebytes == rc);
	if (rc > 0) {
		/* Proceed to next pipe */
		handle->pipe_index = handle->pipe_index->next;
	}	/* if */
	if (rc == 0) {
		/* Get more data from input */
		handle->pipe_index = handle->pipe_index->prev;
	}	/* if */

	return rc;

err1:
	_gpf_free_writebuf(pipe);
err0:
	free(pipe->buf_conv);
	pipe->buf_conv = NULL;
	pipe->buf_conv_size = 0;
	return rc;
}	/* GPF_RGBA_transfernextstreamnextblock */



int GPF_RGBA_getformat(struct gpf_pipeline_item_t *pipe,
			struct gpf_pipeline_item_t *from_pipe,
			int chain)
{
	struct rgbagpf_priv *priv;

	priv = RGBA_PRIV(pipe);

	switch (chain) {
	case INFOLIST_INPUT:
		GPFDPRINT_PIPEMGR("pipe-rgba: get format description for the output\n");

		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT) == NULL);
		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT) != NULL);

		memcpy(&priv->pf_output,
			&(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT)->pf),
			sizeof(struct gpf_pixelformat_t));

		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT)->memory_usage > 0);
		priv->output_data_size =
			PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT)->memory_usage;
		return GGI_OK;

	case INFOLIST_OUTPUT:
		GPFDPRINT_PIPEMGR("pipe-rgba: get format description for the input\n");

		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT) != NULL);
		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_OUTPUT) == NULL);

		memcpy(&priv->pf_input,
			&(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT)->pf),
			sizeof(struct gpf_pixelformat_t));

		LIB_ASSERT(PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT)->memory_usage > 0);
		priv->input_data_size =
			PIPELINEITEM_STRMARK(from_pipe, INFOLIST_INPUT)->memory_usage;
		return GGI_OK;

	}	/* switch */

	return GGI_ENODEVICE;
}	/* GPF_RGBA_getformat */

