//*****************************************************************************
//
// qtremind - an X windows appoint reminder program.
// Copyright (C) 1997  Tom Daley
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// The license is in the file COPYING.
//
// Tom Daley
// 1309 W. Lorraine Ave.
// Woodland Park, CO  80863-2329
//
// tdaley@vsys.com
//
//*****************************************************************************

#include <stream.h>
#include <errno.h>
#include <gdbm.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <qdstream.h>
#include <qregexp.h>

#include "persist.h"

//*****************************************************************************

GDBM_FILE QtedPersist::_dbf = NULL; // HELP, fix this to be a d_array.
int QtedPersist::_dirname_set = 0;
QString QtedPersist::_dir;

//*****************************************************************************

QtedPersist::
QtedPersist() : _key(0), _dirty(1) {
}

//*****************************************************************************

QtedPersist::
QtedPersist(unsigned int key) : _key(key), _dirty(0) {
}

//*****************************************************************************

QtedPersist::
~QtedPersist() {
}

//*****************************************************************************

void QtedPersist::
dbDirty(void) {
   _dirty = 1;
}

//*****************************************************************************

int QtedPersist::
dbIsDirty(void) {
   return (_dirty);
}

//*****************************************************************************

unsigned int QtedPersist::
dbKey(void) {
   return (_key);
}

//*****************************************************************************

int QtedPersist::
dbSetDirectory(QString dir) {
// only allow setting the directory name once.
   int result = 0;

   if (_dirname_set)
      result = -1;
   else {
      _dirname_set = 1;
      _dir = dir.copy();
   }
   return (result);
}

//*****************************************************************************

QString QtedPersist::
dbFileName(QString className) {
   QString file_name;
   file_name = _dir.copy();
   if (file_name.length())
      file_name += "/";
   file_name += className;
   file_name += ".gdbm";

   return (file_name);
}

//*****************************************************************************

int QtedPersist::
dbInitFile(QString className) {
   int result = 0;
   struct stat Buffer;

   if (stat(dbFileName(className), &Buffer)) {	// if file does not exist
      result = dbOpen(className, GDBM_WRCREAT);
      if (result == 0)
         result = -1;
      else 
         dbClose(className);
   }
   return (result);
}

//*****************************************************************************

unsigned int QtedPersist::
dbNextRec(QString className, unsigned int key_num) {
   unsigned int result = 0;
#ifdef SPEW
   cerr << "+ dbNextRec(" << className << ", " << key_num << ")\n";
#endif
   if (dbOpen(className, GDBM_READER)) {
      datum key;
      if (key_num == 0) {
         key = gdbm_firstkey(_dbf);
      } else {
         key.dptr = (char *)&key_num;
         key.dsize = sizeof(key_num);
         key = gdbm_nextkey(_dbf, key);
      }

      while (key.dptr && key.dsize != sizeof(int))	// skip any non-int key
         key = gdbm_nextkey(_dbf, key);

      if (key.dptr) {
         result = *(unsigned int *)key.dptr;
      }
   }
#ifdef SPEW
   cerr << "- dbNextRec(" << result << ")\n";
#endif
   return(result);
}

//*****************************************************************************

int QtedPersist::
dbMkdir(void) {
#ifdef SPEW
   cerr << "+ QtedPersist::dbMkdir()\n";
#endif
   int result = -1;
   if (_dir.length()) {
      struct stat stat_buf;
      if (stat(_dir, &stat_buf) == 0)
         if (S_ISDIR(stat_buf.st_mode))
            result = 0;

// Rip apart the name and try to create directory path
      if (result) {
         int i, index = 0, len;
         QRegExp not_slash("[^/]+");
         result = 0;	// now we'll assume all is well until proven otherwise
         while (index >= 0 && result == 0) {
            QString subdir;
            index = not_slash.match(_dir, index, &len);
            if (index >= 0) {
               subdir = _dir.mid(0, index + len);
               index += len + 1;
#ifdef SPEW
               cerr << "  Directory name is now  " << subdir << "\n";
#endif   
               if (stat(subdir, &stat_buf) < 0) {
                  if (mkdir(subdir, 0755) < 0) {
                     QString errstr;
                     errstr.sprintf("mkdir failed for %s",
                                                     (const char *)subdir);
                     dbError(errstr);	// SYSTEM_ERROR
                     result = -1;
                  }
               } else {
                  if (!S_ISDIR(stat_buf.st_mode)) {
                     QString errstr;
                     errstr.sprintf("file %s exists and is not a directory",
                                                     (const char *)subdir);
                     dbError(errstr); 	 // PERSIST_ERROR
                     result = -1;
                  }
               }
            }
         }
      }
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbMkdir()\n";
#endif
   return(result); 
}

//*****************************************************************************

int QtedPersist::
dbOpen(QString className, int mode) {
   char *cp;
   int result = 1;
   QString file_name;
#ifdef SPEW
   cerr << "+ QtedPersist::dbOpen(" << className << ", " << mode << ")\n";
#endif
   if (_dbf == NULL) {
      if (!_dirname_set) {
         dbError("directory name not set");
         result = 0;
      } else if (className.length() == 0) {
         dbError("className not set");
         result = 0;
      } else if (dbMkdir()) {
         result = 0;
      } else {
//       see if dirname is OK;
         file_name = dbFileName(className);

         _dbf = gdbm_open((char *)(const char *)file_name, 0, mode,
                          0600,0);
         if (_dbf == NULL) {
            QString str;
            str.sprintf("gdbm_open failed for file %s",(const char *)file_name);
            dbError(str);
            result = 0;
         }
      }
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbOpen()\n";
#endif
   return (result);
}

//*****************************************************************************

void QtedPersist::
dbClose(QString className) {
#ifdef SPEW
   cerr << "+ QtedPersist::dbClose(" << className << ")\n";
#endif
   if (_dbf) {
      gdbm_close(_dbf);
      _dbf = NULL;
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbClose()\n";
#endif
}

//*****************************************************************************

void QtedPersist::
dbError(QString str) {
   cerr << str << ":" << gdbm_strerror(gdbm_errno) << ", " << strerror(errno) 
        << "\n";
}

//*****************************************************************************

QList<unsigned int> QtedPersist::
dbFreeList(QString className) {
   QList<unsigned int> list;
#ifdef SPEW
   cerr << "+ QtedPersist::freeList(" << className << ")\n";
#endif
   if (dbOpen(className, GDBM_READER)) {
      datum key, content;
      key.dptr = FREE_LIST;
      key.dsize = strlen(FREE_LIST);
      content = gdbm_fetch(_dbf, key);
      if (content.dptr) {
         QByteArray data;
         data.setRawData(content.dptr, content.dsize);
         QDataStream ds(data, IO_ReadOnly);
         int i, num = content.dsize/sizeof(int);
         for(i = 0; i < num; i++) {
            unsigned int key_num;
            ds >> key_num;
            list.append(new unsigned int(key_num));
         }
         data.resetRawData(content.dptr, content.dsize);
         free(content.dptr); 
      } else {	// there is no free list yet, make one
#ifdef SPEW
         cerr << "There is no free list yet I'll make one\n";
#endif
         int i;
         content.dptr = (char *)&i;
         content.dsize = 0;
         if (gdbm_store(_dbf, key, content, GDBM_REPLACE))
            dbError("Writing first FREE LIST\n");
      }
   }

#ifdef SPEW
   cerr << "- QtedPersist::freeList( ";
   unsigned int *ip;
   for (ip = list.first(); ip; ip = list.next())
      cerr << *ip << " ";
   cerr << ")\n";
#endif   
   return (list);
}

//*****************************************************************************

void QtedPersist::
dbFreeList(QList<unsigned int> list) {
   unsigned int *ip;
#ifdef SPEW
   cerr << "+ QtedPersist::dbFreeList( ";
   for (ip = list.first(); ip; ip = list.next())
      cerr << *ip << " ";
   cerr << ")\n";
#endif
   QByteArray data;
   QDataStream ds(data, IO_WriteOnly);
   for (ip = list.first(); ip; ip = list.next())
      ds << *ip;
   if (dbOpen(dbClassName(), GDBM_WRITER)) {
      int i;
      datum content, key;
      if (data.size())
         content.dptr = data.data();
      else				// if free list empty, we need a dummy
         content.dptr = (char *)&i;	// pointer that is valid 
      content.dsize = data.size();
      key.dptr = FREE_LIST;
      key.dsize = strlen(FREE_LIST);
      if (gdbm_store(_dbf, key, content, GDBM_REPLACE))
         dbError("dbFreeList");
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbFreeList()\n";
#endif
}

//*****************************************************************************

void QtedPersist::
dbShowFreeList(QString className) {
   int was_open = 0;
   if (_dbf)
      was_open = 1;
   else {
      if (dbOpen(className, GDBM_READER)) {
         QList<unsigned int> list;
         list.setAutoDelete(TRUE); 
         list = dbFreeList(className);
    
         cerr << "Free List = ";
         for (unsigned int *ip = list.first(); ip; ip = list.next())
            cerr << *ip << " ";
         cerr << "\n";
 
         if (!was_open)
            dbClose(className);
      }
   }
}

//*****************************************************************************

unsigned int QtedPersist::
dbMaxUsed(QString className) {
   unsigned int result;
#ifdef SPEW
   cerr << "+ QtedPersist::dbMaxUsed(" << className << ")\n";
#endif
   if (dbOpen(className, GDBM_READER)) {
      datum key, content;
      key.dptr = MAX_USED_KEY;
      key.dsize = strlen(MAX_USED_KEY);
      content = gdbm_fetch(_dbf, key);
      if (content.dptr) {
         QByteArray data;
         data.setRawData(content.dptr, content.dsize);
         QDataStream ds(data, IO_ReadOnly);
         ds >> result;               // QByteArray will free the data pointer 
         data.resetRawData(content.dptr, content.dsize);
         free(content.dptr);
      } else {	// there is no free list yet, make one
         result = 0;
         content.dptr = (char *)&result;
         content.dsize = sizeof(result);
         if (gdbm_store(_dbf, key, content, GDBM_REPLACE))
            dbError("Writing first MAX_USED_KEY");
      }
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbMaxUsed(" << result << ")\n";
#endif   
   return (result);
}

//*****************************************************************************

void QtedPersist::
dbMaxUsed(unsigned int new_max) {
   int result;
#ifdef SPEW
   cerr << "+ QtedPersist::dbMaxUsed(" << new_max << ")\n";
#endif
   if (dbOpen(dbClassName(), GDBM_WRITER)) {
      datum key, content;
      QByteArray data;
      QDataStream ds(data, IO_WriteOnly);
      key.dptr = MAX_USED_KEY;
      key.dsize = strlen(MAX_USED_KEY);
      ds << new_max;
      content.dptr = data.data();
      content.dsize = data.size();
      if (gdbm_store(_dbf, key, content, GDBM_REPLACE))
         dbError("Writing MAX_USED_KEY");
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbMaxUsed()\n";
#endif
}

//*****************************************************************************

void QtedPersist::
dbShowMaxUsed(QString className) {
   int was_open = 0;
   if (_dbf)
      was_open = 1;
   else {
      if (dbOpen(className, GDBM_READER)) {
         cerr << "Max Used = " << dbMaxUsed(className) << "\n";
         if (!was_open)
            dbClose(className);
      }
   }
}

//*****************************************************************************

int QtedPersist::
dbSave(QByteArray data) {
   int result = 0;
#ifdef SPEW
   cerr << "+ QtedPersist::dbSave()\n";
#endif
   if (_key == 0 || _dirty) {
      if (dbOpen(dbClassName(), GDBM_WRCREAT)) {
         datum content, key; 
         content.dptr = data.data();
         content.dsize = data.size();
         unsigned int ikey;
         if (_key)
            ikey = _key;
         else
            ikey = dbGetKey();
         key.dptr = (char *)&ikey;
         key.dsize = sizeof(ikey);
         if (gdbm_store(_dbf, key, content, GDBM_REPLACE)) {
            dbError("dbSave");
            result  = -1;
         }
         _key = ikey;
         dbClose(dbClassName());
      }
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbSave()\n";
#endif
   return (result);
}

//*****************************************************************************

unsigned int QtedPersist::
dbGetKey(void) {
   unsigned int *ip, result;
#ifdef SPEW
   cerr << "+ QtedPersist::dbGetKey()\n";
#endif
   QList <unsigned int>free_list; 
   free_list.setAutoDelete(TRUE);	// free values on the list when done
   free_list = dbFreeList(dbClassName());
   if (ip = free_list.first()) {
      result = *ip;
      free_list.removeFirst ();
      dbFreeList(free_list);
   } else {
      result = dbMaxUsed(dbClassName()) + 1;
      dbMaxUsed(result);
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbGetKey(" << result << ")\n";
#endif
   return (result);
}

//*****************************************************************************

int QtedPersist::
dbDelete(void) {
   int result = 0;
#ifdef SPEW
   cerr << "+ QtedPersist::dbDelete()\n";
#endif
   if (_key) {
      if (dbOpen(dbClassName(), GDBM_WRCREAT)) {
         datum key;
         key.dptr = (char *)&_key;
         key.dsize = sizeof(_key);
         if (gdbm_delete(_dbf, key)) {
            dbError("dbDelete");
            result = -1;
         }
         QList<unsigned int> list;
         list.setAutoDelete(TRUE);
         list = dbFreeList(dbClassName());
         list.append(new unsigned int(_key));
         dbFreeList(list); 
         _key = 0;
         dbClose(dbClassName());
      } else {
         result = -1;
      }
   }
#ifdef SPEW
   cerr << "- QtedPersist::dbDelete(" << result << ")\n";
#endif
   return (result);
}

//*****************************************************************************

QByteArray *QtedPersist::
dbGet(QString className, unsigned int key_num) {
   QByteArray *result = NULL;
   if (dbOpen(className, GDBM_READER)) {
      datum key, content;
      key.dptr = (char *)&key_num;
      key.dsize = sizeof(key_num);
      content = gdbm_fetch(_dbf, key);
      if (content.dptr) {
         result = new QByteArray;
         result->duplicate(content.dptr, content.dsize);
         free(content.dptr);
      } else {
         cerr << "Could not find record\n";
      } 
   }
   return (result);
}

//*****************************************************************************
